/**************************************************************************************/
/* ACODTSP.c                      Version 1.0                                         */
/* This is an implementation of the RIACO, EIACO, HIACO and MIACO algorithms proposed */
/* for the dynamic travelling salesman problem (DTSP) of the following paper:         */
/*                                                                                    */
/*   M. Mavrovouniotis and S. Yang, Ant colony optimization with immigrants schemes   */
/*   for the dynamic travelling salesman problem with traffic factors. Applied Soft   */
/*   Computing, Elsevier, published online 10 June, 2013                              */
/*                                                                                    */
/* Compile:                                                                           */
/*   g++ -o ACODTSP ACODTSP.c                                                         */
/* Run:                                                                               */
/*   ./ACODTSP problem_instance change_mode change_degree change_speed                */
/*                                                                                    */
/*   e.g., ./ACODTSP eil51.tsp 1 0.25 100                                             */
/*                                                                                    */
/* Written by:                                                                        */
/*   Michalis Mavrovouniotis, De Montfort University, UK; July 2013                   */
/*                                                                                    */
/* If any query, please email Michalis Mavrovouniotis at mmavrovouniotis@dmu.ac.uk    */ 
/*                                                                                    */
/*  Some ACO methods are based on the original ACO framework implementation:          */ 
/*                                                                                    */
/*    Thomas Stuetzle. ACOTSP, Version 1.02. Available from                           */
/*    http://www.aco-metaheuristic.org/aco-code, 2004.                                */           
/*                                                                                    */
/**************************************************************************************/

#include<iostream>
#include<stdio.h>
#include<stdlib.h>
#include<string>
#include<cstring>
#include<math.h>
#include<fstream>
#include<cmath>
#include<time.h>
#include<limits.h>

using namespace std;

/*------------------DTSP with Traffic Factors Implementation-------------------------*/
#define EPSILON  0.000000000000000000000001
#define INFTY    INT_MAX
#define CHAR_LEN 100

struct object { 
  int id;
  double x;
  double y;
};

struct env_struct {
 int id;
 double **base;
};

struct object *init_objects;     //Actual city list of the instance
struct env_struct *cyclic_bases; //Store enviornments for cyclic
int **distances;                 //Distance matrix
double **traffic_factors;        //Traffic factors matrix 
double lambda_l = 1;             //Lower bound of traffic
double lambda_u = 5;             //Upper bound of traffic

int problem_size;                //Size of the instance
char* problem_instance;          //Name of the instance
int change_mode;                 //How environment changes, reappear or not
                                 //1:random, 2:cyclic, 3:cyclic with noise, 4:varying
double change_degree;            //Degree of changes
int change_speed;                //Peiord of changes in algorithmic iterations
int cyclic_states = 4;           //Base states for environments that reappaear
double p_noise = 0.05;           //Noise probability for environments that reappear
int cyclic_base_count;           //Base states index for cyclical reappeared environments

int current_iteration;           //Used to calculate the period of change
int seed;                        
int max_iterations;
int max_trials;


//Used to output offline performance and population diversity
int** performance;
double** diversity;
FILE *log_performance;
FILE *log_diversity;
FILE *log_varying_values;

/****************************************************************/
/*Random number generator where the seed is the same in all runs*/
/****************************************************************/
double env_random_number(double low, double high) {
  return ((double)(rand()%10000)/10000.0)*(high - low) + low;  
}

/****************************************************************/
/*           Initialize all traffic factors equally             */
/****************************************************************/
void initialize_traffic_factors(){
  int i,j;
  for(i = 0; i < problem_size; i++){
   for(j = 0; j <= i; j++) {
     traffic_factors[i][j] = 1.0;
     traffic_factors[j][i] = traffic_factors[i][j];
   }
  }
}

/****************************************************************/
/*Compute and return the euclidean distance of two cities       */
/****************************************************************/
int euclidean_distance(int i, int j) {
  double xd,yd;
  int r = 0;
  xd = init_objects[i].x - init_objects[j].x;
  yd = init_objects[i].y - init_objects[j].y;
  r  = sqrt(xd*xd + yd*yd) + 0.5; 
  return r;
}

/****************************************************************/
/*Compute the distance matrix of the problem instance           */
/****************************************************************/
void compute_distances(void) {
  for(int i = 0; i < problem_size; i++){
    for(int j = 0; j < problem_size; j++){
      distances[i][j] = (int)euclidean_distance(i,j) * traffic_factors[i][j];    
    }
  }
}

/****************************************************************/
/* Perform random changes by changing the distances between     */
/* cities                                                       */
/****************************************************************/
void add_random_change(void){
  int i,j; 
  for(i = 0; i < problem_size; i++){ 
    for(j = 0; j < problem_size; j++){ 
      if(env_random_number(0.0,1.0) <= change_degree){ 
        traffic_factors[i][j] = 1.0 + env_random_number(lambda_l, lambda_u);
        traffic_factors[j][i] = 1.0 + env_random_number(lambda_l, lambda_u);
      } else {
        traffic_factors[i][j] = 1.0; 
        traffic_factors[j][i] = 1.0;
      }
    }
  }
}

/****************************************************************/
/*Generate and return a two-dimension array of type int         */
/****************************************************************/
int ** generate_2D_matrix_int(int n, int m){
  int **matrix;
  matrix = new int*[n];
  for ( int i = 0 ; i < n ; i++ ) {
    matrix[i] = new int[m];
  }
  //initialize 2-d array
  for(int i = 0; i < n; i++){
    for(int j = 0; j < m; j++) {
      matrix[i][j] = 0;
    }
  }
  return matrix;
}

/****************************************************************/
/*Generate and return a two-dimension array of type double      */
/****************************************************************/
double ** generate_2D_matrix_double(int n, int m){
  double **matrix;

  matrix = new double*[n];
  for ( int i = 0 ; i < n ; i++ ) {
    matrix[i] = new double[m];
  }
  //initialize the 2-d array
  for(int i = 0; i < n; i++){
    for(int j = 0; j < m; j++) {
      matrix[i][j] = 0.0;
    }
  }
  return matrix;
}

/****************************************************************/
/* Construct the base state for the environments that reappear  */
/* Note that this method uses the cyclic_states global variable */    
/* to generate and store a number of environments               */
/****************************************************************/
void generate_cyclic_environment(){
  int i, j, h;
  cyclic_bases = new env_struct[cyclic_states];
  for(h = 0; h < cyclic_states; h++){
    cyclic_bases[h].id = h;
    cyclic_bases[h].base = generate_2D_matrix_double(problem_size,problem_size);
    for(i = 0; i < problem_size; i++) {
      for(j = 0; j < problem_size; j++) {
         if(env_random_number(0.0,1.0) <= change_degree){ 
           cyclic_bases[h].base[i][j] = 1.0 + env_random_number(lambda_l,lambda_u);
           cyclic_bases[h].base[j][i] = 1.0 + env_random_number(lambda_l,lambda_u);
         } else {
           cyclic_bases[h].base[i][j] = 1.0; 
           cyclic_bases[h].base[j][i] = 1.0;
         }
       }
     }
   }
}

/****************************************************************/
/* Perfom an indexed, using the state variable, change based on */
/* the environments generated in the generate_cyclic_environment*/ 
/****************************************************************/
void add_cyclic_change(int state){
  int i, j;
  double noise;
  for(i = 0; i < problem_size; i++) { 
   for(j = 0; j < problem_size; j++) { 
    if(change_mode == 2) {  //cyclic 
     traffic_factors[i][j] = cyclic_bases[state].base[i][j];
     traffic_factors[j][i] = cyclic_bases[state].base[j][i];
    } 
    if(change_mode == 3) {
      if(env_random_number(0.0,1.0) <= p_noise)
        noise = env_random_number(0.0,1.0);
      else
        noise = 0.0;
      traffic_factors[i][j] = cyclic_bases[state].base[i][j] + noise;
      traffic_factors[j][i] = cyclic_bases[state].base[j][i] + noise;
      }
    }
  }
}

/****************************************************************/
/* Perfom a dynamic change with randomly generated change_speed */
/* and random change_degree.                                    */
/****************************************************************/
void varying_environmental_changes(){
  int i, j;
  change_speed = (int)env_random_number(1,100);//random number 1..100
  change_degree = env_random_number(0.0,1.0);  //random number 0..1
  for(i =0; i < problem_size; i++){ 
   for(j = 0; j < problem_size; j++){ 
      if(env_random_number(0.0,1.0) <= change_degree){ 
        traffic_factors[i][j] = 1.0 + env_random_number(lambda_l, lambda_u);
        traffic_factors[j][i] = 1.0 + env_random_number(lambda_l, lambda_u);
      } else {
        traffic_factors[i][j] = 1.0; 
        traffic_factors[j][i] = 1.0;
      }
   }
  }
}


/****************************************************************/
/* Read the problem instance and generate the city list         */
/****************************************************************/
void read_problem(char *filename){
  char line[CHAR_LEN];
  char * keywords;
  char Delimiters[] = " :=\n\t\r\f\v";
  ifstream fin(filename);
  while((fin.getline(line, CHAR_LEN-1))){
    if(!(keywords = strtok(line, Delimiters)))
      continue;
    //keywords = strupr(keywords);
    if(!strcmp(keywords, "DIMENSION")){			
      if(!sscanf(strtok(NULL, Delimiters), "%d", &problem_size)){
	cout<<"DIMENSION error"<<endl;
	exit(0);
      }
    }
    else if(!strcmp(keywords, "EDGE_WEIGHT_TYPE")){
      char * tempChar;
      if(!(tempChar=strtok(NULL, Delimiters))){
	cout<<"EDGE_WEIGHT_TYPE error"<<endl;
	exit(0);
      }
      if(strcmp(tempChar, "EUC_2D")){
	cout<<"not EUC_2D"<<endl;
	exit(0);
      }
    }
    else if(!strcmp(keywords, "NODE_COORD_SECTION")){
      init_objects = new object[problem_size];
      int i;
      for(i=0; i<problem_size; i++){
	//store initial cities
        fin>>init_objects[i].id;       
	fin>>init_objects[i].x>>init_objects[i].y;
        init_objects[i].id -=1;
      }
    }
  }
  fin.close();
}

/****************************************************************/
/* Initialize the environment and perform the initial dynamic   */
/*  change/generate base states                                 */
/****************************************************************/
void initialize_environment(){
  //initialize distances with traffic factors
  distances = generate_2D_matrix_int(problem_size,problem_size);
  traffic_factors = generate_2D_matrix_double(problem_size,problem_size);
  initialize_traffic_factors();
  switch(change_mode) {
  case 1: //random environment
    add_random_change();
    break;
  case 2://cylic
  case 3://cyclic with noise
    generate_cyclic_environment();
    cyclic_base_count = 0;
    add_cyclic_change(cyclic_base_count);
    break;
  case 4: //varying environments
    change_speed = 10; 
    change_degree = 0.1;
    varying_environmental_changes;
    break;
  }
  compute_distances();
}

/****************************************************************/
/* Perform the dynamic change every "period" iteration          */
/****************************************************************/
void change_environment(){ 
   switch(change_mode){ 
    case 1: // random
      add_random_change();
      break;
    case 2: //cyclic
    case 3: //cyclic with noise  
       //cycle on the base states as a fixed logical ring
      if(cyclic_base_count == cyclic_states-1)
	cyclic_base_count=0;
      else
	cyclic_base_count++;
      add_cyclic_change(cyclic_base_count);
      break;
    case 4: // varying
      varying_environmental_changes();
      break;          
    }  
    compute_distances(); 
}

/****************************************************************/
/* Validate the values input by the user                        */
/****************************************************************/
void check_input_parameters() {

 if(problem_size == 0) {
   cout << "wrong problem instance file" << endl;
   exit(2);
 }
 if(change_mode > 4 || change_mode <= 0) {
   cout << "select a valid change mode (between 1..4)" << endl;
   exit(2);
 }
 if(change_degree > 1.0 || change_degree <= 0.0) {
   cout << "select a valid change degree (between 0..1)" << endl;
   exit(2);
 }
 if(change_speed <= 0) {
   cout << "select a valid change speed (int)" << endl;
   exit(2);
 }
}

/****************************************************************/
/* Evaluate a TSP tour and return the length. This method can   */
/* be used in the optimizer integrated with DTSP. The TSP tour  */
/* to input is an array of integers, where each integer         */
/* corresponds to a city index                                  */
/****************************************************************/
int fitness_evaluation(int *t) {
  int i, tour_length = 0;
  for (i = 0 ; i < problem_size; i++ ) {
    tour_length += distances[t[i]][t[i+1]];
  }
  return tour_length;
}
/*------------------------End of DTSP---------------------------*/

/*------------------ACO Implementations-------------------------*/

#define IA       16807
#define IM       2147483647
#define AM       (1.0/IM)
#define IQ       127773
#define IR       2836
#define MASK     123459876

//Ant or individual structure that represent the TSP tour and tour cost
struct ant{
  int *tour;
  bool *visited;
  int tour_length;
};

int alg_mode = 1;             //1: RIACO, 2: EIACO, 3: HIACO, 4: MIACO

struct ant *ant_population;   //Population of ants or individuals
ant *best_so_far_ant;         //current best so far tour 
ant *previous_best_so_far_ant;//current-1 best so far tour 
struct ant *previous_best;

double **pheromone;          //Pheromone matrix
double **heuristic;          //Heuristic information matrix
double **total;              //Pheromone + Heuristic information matrix

double *prob_of_selection;  //Selection probabilities

//General ACO parameters 
double alpha;
double beta;
double q_0;
double trail0;

int n_ants;                //Population size
int depth;                 //Candidate list size (nearest neighbour)
int **nn_list;             //Candidate lists

struct ant *short_memory; //Short memory for RIACO,EIACO,HIACO and MIACO
struct ant *long_memory;  //Long  memory for MIACO

double immigrant_rate;    //Immigrants replacement ratio
double p_mi;              //Immigrants mutation probability(EIACO,HIACO,MIACO)

int short_memory_size;    //Short memory size

int long_memory_size;     //Long memory size
int t_m;                  //Long memory dynamic update
bool *random_point;       //Initialization with random points

//output files
char * perf_filename; 
char * div_filename; 
char * var_filename;

/****************************************************************/
/*                     Initialization                           */
/****************************************************************/
void set_algorithm_parameters(void){
  //change the algorithm parameters according to your needs
  alpha = 1;
  beta = 5;
  q_0 = 0.0;
  n_ants = 50;
  depth = 20;
  short_memory_size = 6;
  immigrant_rate = 0.4;
  p_mi = 0.01;
  long_memory_size = 3;
}

void allocate_ants(void){
  int i;  

  ant_population = new ant[n_ants];
  for(i = 0; i < n_ants; i++){
    ant_population[i].tour = new int[problem_size+1];
    ant_population[i].visited = new bool[problem_size];
  }

  best_so_far_ant = new ant;
  best_so_far_ant->tour = new int[problem_size+1];
  best_so_far_ant->visited = new bool[problem_size];
  
  prob_of_selection = new double[depth +1];

  if(alg_mode!=1){
    //track the best solution of the previous environment
    //used in EIACO, HIACO and MIACO
    previous_best = new ant[2];
    for(i = 0; i < 2; i++){
      previous_best[i].tour = new int[problem_size+1];
      previous_best[i].visited = new bool[problem_size];
    }  
    previous_best_so_far_ant = new ant;
    previous_best_so_far_ant->tour = new int[problem_size+1];
    previous_best_so_far_ant->visited = new bool[problem_size]; 
  }
} 

void allocate_structures(void){
  int i;

  pheromone = generate_2D_matrix_double(problem_size,problem_size);
  heuristic = generate_2D_matrix_double(problem_size,problem_size);
  total = generate_2D_matrix_double(problem_size,problem_size);
  nn_list = generate_2D_matrix_int(problem_size,depth);
  short_memory = new ant[short_memory_size];
  if(alg_mode==4){ //initialize long memory for MIACO
    long_memory= new ant[long_memory_size];
    random_point = new bool[long_memory_size];
  }
}

void swap(int v[], int v2[],  int i, int j){
  int tmp;
  
  tmp = v[i];
  v[i] = v[j];
  v[j] = tmp;
  tmp = v2[i];
  v2[i] = v2[j];
  v2[j] = tmp;
}

void sort(int v[], int v2[], int left, int right){
  int k, last;

  if (left >= right) 
    return;
  swap(v, v2, left, (left + right)/2);
  last = left;
  for (k=left+1; k <= right; k++)
    if (v[k] < v[left])
      swap(v, v2, ++last, k);
  swap(v, v2, left, last);
  sort(v, v2, left, last);
  sort(v, v2, last+1, right);
}

void compute_nn_lists( void ){
  int i,j;
  int *distance_vector;
  int *help_vector;

  distance_vector = new int[problem_size];
  help_vector = new int[problem_size];
  //compute the nearest neigbhours of the cities
  for (j  = 0 ; j < problem_size; j++ ) { 
    for (i = 0 ; i < problem_size ; i++ ) { 
      distance_vector[i] = distances[j][i];
      help_vector[i] = i;
    }
    distance_vector[j] = INT_MAX;
    sort(distance_vector, help_vector, 0, problem_size-1);
    for (i = 0 ; i < depth ; i++ ) {
      nn_list[j][i] = help_vector[i];
    }
  }

  //free memory
  delete[] distance_vector;
  delete[] help_vector;
}

void init_pheromone_trails(double initial_trail){
  int i,j;
  
  for(i = 0; i < problem_size; i++){
    for(j = 0; j <=i; j++){
      pheromone[i][j] = initial_trail;
      pheromone[j][i] = initial_trail;
    }
  } 
}

void init_heuristic_info(void){
  int i,j;
  
  for(i = 0; i <problem_size; i++){
    for(j = 0; j <=i; j++){
      heuristic[i][j] = 1.0/(double)(distances[i][j] + EPSILON); //small value to avoid 1 div 0
      heuristic[j][i] = heuristic[i][j];
    }
  }
}

void compute_total_info(void){
  int i,j;
  
  for(i = 0; i < problem_size; i++){
    for(j = 0; j < i; j++){
      total[i][j] = pow(pheromone[i][j],alpha) * pow(heuristic[i][j],beta);
      total[j][i] = total[i][j];
    }
  }
}

/****************************************************************/
/*                    Construct Solutions                       */
/****************************************************************/
void ant_empty_memory(ant *a){
  int i;
  //clear previous ant solution
  for(i =0; i < problem_size; i++){
    a->visited[i] = false;
  }
}

double alg_random_number(int *idum){
  int k;
  double ans;
  //uniformly distributed random number [0,1]
  k =(*idum)/IQ;
  *idum = IA * (*idum - k * IQ) - IR * k;
  if (*idum < 0 ) *idum += IM;
  ans = AM * (*idum);
  return ans;
}

void place_ant(ant *a, int step){
  int rnd;
  //place ants to randomly selected cities
  rnd = (int)(alg_random_number(&seed) * (double)problem_size);
  a->tour[step] = rnd;
  a->visited[rnd] = true;
}

void choose_best_next(ant *a, int phase){
  int i,current, next;
  double value_best;

  next = problem_size;
  current = a->tour[phase-1]; //current city of ant
  value_best = -1.0;  //values in the list are always >=0.0
  //choose the next city with maximal (pheromone+heuristic) value 
  //among all cities 
  for(i = 0; i < problem_size; i++){
    if(a->visited[i] == false){
      //if not visited
      if(total[current][i] > value_best){
	next = i;
	value_best = total[current][i];
      }
    }
  }
  a->tour[phase] = next;
  a->visited[next] = true;
}

void neighbour_choose_best_next(ant *a, int phase){
  int i,current,next,temp;
  double value_best, help;
  
  next = problem_size;
  current = a->tour[phase-1]; //current city of ant
  value_best = -1.0; //values in the list are always >=0.0
  //choose the next city with maximal (pheromone+heuristic) value 
  //among all the nearest neighbour cities
  for(i =0; i < depth; i++){
    temp = nn_list[current][i];
    if(a->visited[temp] == false){
      //if not visited
      help = total[current][temp];
      if(help > value_best){
	value_best=help;
	next = temp;
      }
    }
  }
  if(next == problem_size){ 
    //if all nearest neighnour cities are already visited
    choose_best_next(a,phase);
  }else {
    a->tour[phase] = next;
    a->visited[next] = true;    
  }
}

void neighbour_choose_and_move_to_next(ant *a, int phase){
  int i,help, current, select;
  double rnd;
  double partial_sum = 0.0;
  double sum_prob = 0.0;
  double *prob_ptr;
  
  if((q_0 > 0.0) && (alg_random_number(&seed) < q_0)) {
    //with probability q_0 make the best possible choice
    neighbour_choose_best_next(a,phase);
    return;
  }
  
  prob_ptr = prob_of_selection; //selection probabilities of the nearest neigbhour cities
  current = a->tour[phase-1]; //current city
  //compute selection probabilities of nearest neigbhour cities
  for(i = 0; i < depth; i++){
    if(a->visited[nn_list[current][i]]){
      prob_ptr[i] = 0.0;
    }else{
      prob_ptr[i] = total[current][nn_list[current][i]];
      sum_prob +=prob_ptr[i];
    }
  }
  if(sum_prob <=0.0){
    //in case all neigbhbour cities are visited
    choose_best_next(a,phase);
  } else{
    //proabilistic selection (roullete wheel) 
    rnd = alg_random_number(&seed);
    rnd *= sum_prob;
    select = 0;
    partial_sum = prob_ptr[select];
    while(partial_sum<=rnd){
      select++;
      partial_sum+=prob_ptr[select];
    }
    //this may very rarely happen because of rounding if 
    //rnd is close to 1
    if(select==depth){
      neighbour_choose_best_next(a,phase);
      return;
    }
    help = nn_list[current][select];
    a->tour[phase] = help;
    a->visited[help] = true;
  }
}

void construct_solutions(void){
  int k,step;
  //clear memory of ants
  for(k =0; k < n_ants; k++){
    ant_empty_memory(&ant_population[k]);
  }
  step = 0; 
  //place ants on a random city
  for(k =0; k < n_ants; k++){
    place_ant(&ant_population[k],step);
  }
  //select next city until all are visited
  while(step < problem_size-1){
    step++;
    for(k =0; k < n_ants; k++){
      neighbour_choose_and_move_to_next(&ant_population[k],step);
    }
  }
  step = problem_size;
  for(k =0; k < n_ants; k++){
    //close TSP tour, i.e., the first city needs to be identical with the last one.
    ant_population[k].tour[problem_size] = ant_population[k].tour[0];
    ant_population[k].tour_length = fitness_evaluation(ant_population[k].tour);//evalute 
  }
}

void choose_closest_next(ant *a, int phase){
  int i,current,next,min;  
  next = problem_size;
  current = a->tour[phase-1]; //current city of ant
  min = INFTY;
  //choose closest city used in the nn_tour()
  for(i = 0; i < problem_size; i++){
    if(a->visited[i] == false){
      //if not visited
      if(distances[current][i] < min){
	next = i;
	min = distances[current][i];
      }
    } 
  }
  a->tour[phase] = next;
  a->visited[next] = true;
}

int nn_tour(void){
  int phase, help; 
  phase=help=0;  
  ant_empty_memory(&ant_population[0]);
  place_ant(&ant_population[0],phase);
  //compute the tour length of the nearest neigbour heuristic
  //used to initialize the pheromone trails
  while(phase < problem_size -1){
    phase++;
    choose_closest_next(&ant_population[0],phase);
  }
  phase = problem_size;
  ant_population[0].tour[problem_size] = ant_population[0].tour[0];
  ant_population[0].tour_length = fitness_evaluation(ant_population[0].tour);
  help = ant_population[0].tour_length;
  ant_empty_memory(&ant_population[0]);

  return help;
}


/****************************************************************/
/*                    Generate Immigrants                       */
/****************************************************************/
int *generate_random_immigrant(){
  int *random_immigrant;
  int i,help, object, tot_assigned =0;
  int *r;

  random_immigrant = new int[problem_size+1];  
  r = new int[problem_size];
  //set indexes of cities
  for(i = 0; i < problem_size; i++){
    r[i]=i;
  } 
  //randomly change indexes of cities
  for(i = 0; i < problem_size; i++){
    object = (int) (alg_random_number(&seed) * (double)(problem_size-tot_assigned));
    help = r[i];
    r[i]=r[i+object];
    r[i+object]=help;
    tot_assigned++;
  }
  //generate random immigrant
  for(int i = 0; i < problem_size; i++){
    random_immigrant[i]=r[i];
  }
  //close TSP tour to represent a feasible solution
  random_immigrant[problem_size] = random_immigrant[0];

  //free memory
  delete[] r;

  return random_immigrant;
}

int *generate_elitism_based_immigrant(){
  int *elitism_immigrant;
  int i,help,object;
  
  elitism_immigrant = new int[problem_size+1];
  //re-evaluate/repair the best ant from the previous environment
  previous_best_so_far_ant->tour_length = fitness_evaluation(previous_best_so_far_ant->tour);
  //copy the previous best ant
  for(i =0; i < problem_size; i++){
    elitism_immigrant[i] = previous_best_so_far_ant->tour[i];
  }
  for(i= 0; i < problem_size; i++){
    //perform random swaps according to p_mi (small)
    if(alg_random_number(&seed) <= p_mi){
      help =0;
      object = (int)(alg_random_number(&seed) * (double)problem_size);
      help = elitism_immigrant[i];
      elitism_immigrant[i] = elitism_immigrant[object];
      elitism_immigrant[object] = help;
    }
  }
  //close TSP tour to represent a feasible solution
  elitism_immigrant[problem_size] = elitism_immigrant[0];
 
  return elitism_immigrant;
}

int find_memory_best(void){
  int k,min,k_min; 
  min = long_memory[0].tour_length;
  k_min = 0; 
  for(k =1; k < long_memory_size; k++){
    if(long_memory[k].tour_length < min) {
      min=long_memory[k].tour_length;
      k_min = k;
    }
  }
  return k_min; //memory-best ant index
}

int *generate_memory_based_immigrant(){
  int *memory_immigrant;
  int i,help,object,mem_ind;
  
  memory_immigrant = new int[problem_size+1];
  mem_ind = find_memory_best(); //retrieve best from long memory

  for(i =0; i < problem_size; i++){
    memory_immigrant[i] = long_memory[mem_ind].tour[i];
  } 
  for(i=0; i < problem_size; i++){
    //perform random swaps according to p_mi (small)
    if(alg_random_number(&seed) <= p_mi){
      help = 0;
      object = (int)(alg_random_number(&seed) * (double)problem_size);
      help = memory_immigrant[i];
      memory_immigrant[i] = memory_immigrant[object];
      memory_immigrant[object] = help;
    }
  }
  //close TSP tour to represent a feasible solution
  memory_immigrant[problem_size] = memory_immigrant[0];
  
  return memory_immigrant;
}


/****************************************************************/
/*                   Update Short-term Memory                    */
/****************************************************************/
void quick_swap(int v[], int v2[], int i, int j){
  int tmp;
  tmp = v[i];
  v[i] = v[j];
  v[j] = tmp;
  tmp = v2[i];
  v2[i] = v2[j];
  v2[j] = tmp;
}

void quick_sort(int v[], int v2[], int left, int right){
  int k,last;  
  if (left >=right)
    return;
  quick_swap(v,v2,left, (left+right)/2);
  last = left;
  for(k = left+1; k <= right; k++)
    if(v[k]<v[left])
      quick_swap(v,v2,++last,k);
  quick_swap(v,v2,left,last);
  quick_sort(v,v2,left,last);
  quick_sort(v,v2,last+1,right);
}

void copy_from_to(ant *a1, ant *a2){
  int i; 
  //ant2 is a copy of ant1
  a2->tour_length = a1->tour_length;
  for(i = 0; i < problem_size; i++){
    a2->tour[i]=a1->tour[i];
  }
  a2->tour[problem_size] = a2->tour[0];
}

void update_short_term_memory(){
  int i,im_size;
  int *tours = new int[n_ants];
  int *id = new int[n_ants];
  //number of immigrants
  im_size = (int)trunc(short_memory_size * immigrant_rate);
  ant *immigrants = new ant[im_size];  

  for(i = 0; i < im_size; i++){
    switch(alg_mode){
    case 1: //RIACO    
      immigrants[i].tour = generate_random_immigrant();
      immigrants[i].tour_length = fitness_evaluation(immigrants[i].tour);       
      break;
    case 2: //EIACO
      immigrants[i].tour = generate_elitism_based_immigrant();
      immigrants[i].tour_length = fitness_evaluation(immigrants[i].tour);    
      break;
    case 3: //HIACO
      if (i % 2 == 0)
       immigrants[i].tour = generate_random_immigrant();
      else 
       immigrants[i].tour = generate_elitism_based_immigrant();
       immigrants[i].tour_length = fitness_evaluation(immigrants[i].tour); 
      break;
    case 4: //MIACO
      immigrants[i].tour = generate_memory_based_immigrant();
      immigrants[i].tour_length = fitness_evaluation(immigrants[i].tour); 
     break;
    }
  }
  //add new generation of the best ants to short memory
  for(i = 0; i < n_ants; i++){
    tours[i] = ant_population[i].tour_length;
    id[i] = i;
  }
  quick_sort(tours,id,0,n_ants-1);
  for(i = 0; i < short_memory_size; i++){
    short_memory[i] = ant_population[id[i]];
  }
  //replace immigrants with the worst ants
  for(i = short_memory_size - 1; i > short_memory_size - im_size - 1; i--){
    copy_from_to(&immigrants[short_memory_size-1-i], &short_memory[i]);
  }
  //free memory
  delete[] tours;
  delete[] id; 
   for(int i =0; i < im_size; i++)
   delete[] immigrants[i].tour;
   delete[] immigrants;
}

/****************************************************************/
/*                    Update Long-term Memory                   */
/****************************************************************/
void init_memory_randomly(void){
  int i;
  for(i = 0; i < long_memory_size; i++){
    long_memory[i].tour = generate_random_immigrant(); //random tours
    long_memory[i].tour_length = fitness_evaluation(long_memory[i].tour);
    random_point[i] = true;
  } 
  t_m = 5 + ((int)(alg_random_number(&seed) * 6.0)); //random number 5..10
}

bool detect_change(){
  int i,total_before, total_after;
  bool flag = false;
  
  total_before=total_after=0;
  //calculate total cost before dynamic change
  for (i = 0; i < long_memory_size; i++){
    total_before += long_memory[i].tour_length;
  }
  //calculate total cost after dynamic change
  for(i = 0; i < long_memory_size; i++){
    long_memory[i].tour_length = fitness_evaluation(long_memory[i].tour);
    total_after += long_memory[i].tour_length;
  }
  //is different then a dynamic change occured
  if(total_after == total_before)
    flag = false;
  else 
    flag= true;  
  
  return flag;
}

double distance_between( ant *a1, ant *a2){
  int i, j, h, pos, pred;
  int distance;
  int *pos2 = new int[problem_size];   
  //indexes of cities of ant2
  for (int i = 0 ; i < problem_size ; i++ ) {
    pos2[a2->tour[i]] = i;
  }  
  distance = 0;
  for (i = 0 ; i < problem_size ; i++ ) {
    j = a1->tour[i];
    h = a1->tour[i+1];
    pos = pos2[j];
    if (pos - 1 < 0)
      pred = problem_size - 1;
    else 
      pred = pos - 1;
    if (a2->tour[pos+1] == h)
      distance++;  //common edges
    else if (a2->tour[pred] == h)
      distance++; //common edges
  }  
   //free memory
  delete[] pos2; 

  //     1 - (common_edges/problem size)
  return 1.0 - (distance/(double)problem_size);
}

void update_memory_every_change() {
 int i, index, closest_ind;
 double d, closest = INFTY; 
 
 index=closest_ind= -1;
 //if random point still exist, i.e., if index = -1; then replace
 for(i = 0; i < long_memory_size; i++) {
   if (random_point[i] == true){
     index = i; 
     random_point[i] = false;
     break;
   }
 }
 //re-evaluate/repair the best ant from the previous environment
 previous_best_so_far_ant->tour_length = fitness_evaluation(previous_best_so_far_ant->tour);
 if(index != -1) {
   copy_from_to(previous_best_so_far_ant, &long_memory[index]);
 }else {
   for(i = 0; i < long_memory_size; i++){
     //find the closest ant of the memory with the previous best ant
     d = distance_between(previous_best_so_far_ant, &long_memory[i]);
     if(closest > d) {
       closest = d;
       closest_ind = i;
     }
   }
   //replace
   if(previous_best_so_far_ant->tour_length < long_memory[closest_ind].tour_length){ 
     copy_from_to(previous_best_so_far_ant, &long_memory[closest_ind]);
   }
 }
}

void update_memory_dynamically(){
 int i, index, closest_ind;
 double d, closest=INFTY; 
 
 index=closest_ind= -1;
 //if random point still exist, i.e., if index = -1; then replace
 for(i = 0; i < long_memory_size; i++){ 
   if(random_point[i] == true) { 
     index = i; 
     random_point[i] = false;
     break;
   }
 }
 if(index != -1) {
   copy_from_to(best_so_far_ant, &long_memory[index]);
 }else {
   for(i = 0; i < long_memory_size; i++){
     //find the closest ant of the memory with the current best ant
     d = distance_between(best_so_far_ant, &long_memory[i]);
     if(closest > d) {
       closest = d;
       closest_ind = i;
     }
   }
   //replace
   if(best_so_far_ant->tour_length < long_memory[closest_ind].tour_length){ 
     copy_from_to(best_so_far_ant, &long_memory[closest_ind]);
   }
 }
}
 
void update_long_term_memory(int t){
  int rnd; 
  bool flag = detect_change();  
  if(flag == true){       
    update_memory_every_change();
  }
  if(t == t_m && flag == false){ 
    update_memory_dynamically();
    rnd = 5 + ((int)(alg_random_number(&seed) * 6.0));//random number 5..10
    t_m = t + rnd;   
 }
 //if both cases are true give priority to 
 //update_memory_every_change(), but update the 
 //time for the next update of update_memory_dynamically();
  if(t == t_m && flag == true) {
    rnd = 5 + ((int)(alg_random_number(&seed) * 6.0));//random number 5..10
    t_m = t + rnd;   
  }
}

/****************************************************************/
/*                    Pheromone Update                          */
/****************************************************************/
void constant_pheromone_deposit(ant *a, double deltaT){
  int i,j,h;
  
  for(i = 0; i < problem_size; i++){
    j = a->tour[i];
    h = a->tour[i+1];
    pheromone[j][h]+= deltaT;
    pheromone[h][j] = pheromone[j][h];
  }
}

void generate_pheromone_matrix(void){
  int i;
  double deltaT; //amount of pheromone
  deltaT = (1.0 - trail0) / (double) short_memory_size;
  init_pheromone_trails(trail0);
  for(i = 0; i < short_memory_size; i++){  
    //update pheromone using the ants stored in short memory
    constant_pheromone_deposit(&short_memory[i], deltaT);
  }
}

void pheromone_update(){
  if(alg_mode == 1 || alg_mode == 2 || alg_mode == 3 || alg_mode ==4){
    if(alg_mode == 4) update_long_term_memory(current_iteration); //MIACO
    update_short_term_memory();
    generate_pheromone_matrix();
  }
  compute_total_info(); //heuristic info + pheromone trails
}

/****************************************************************/
/*                    Update Best Ants                          */
/****************************************************************/
int find_best(void){
  int k,min,k_min; 
  min = ant_population[0].tour_length;
  k_min = 0; 
  for(k = 1; k < n_ants; k++){
    if(ant_population[k].tour_length < min) {
      min=ant_population[k].tour_length;
      k_min = k;
    }
  }
  return k_min; //population best ant index
}

void update_best(void){
  int iteration_best = find_best(); 
  if(ant_population[iteration_best].tour_length < best_so_far_ant->tour_length) {
    copy_from_to(&ant_population[iteration_best],best_so_far_ant);
  }
  if(alg_mode!=1){ //RIACO does not transfer any knowledge from previous environments
     copy_from_to(best_so_far_ant, previous_best_so_far_ant); 
     copy_from_to(&previous_best[1], &previous_best[0]); 
     copy_from_to(previous_best_so_far_ant, &previous_best[1]);
     copy_from_to(&previous_best[0],previous_best_so_far_ant);
     if(current_iteration==1) copy_from_to(best_so_far_ant, previous_best_so_far_ant);
   } 
}

/*---------------------------End of ACO-------------------------*/

/****************************************************************/
/*                Update Statistics and Output                  */
/****************************************************************/
void open_stats(void){
  performance = generate_2D_matrix_int(max_trials,max_iterations);
  diversity = generate_2D_matrix_double(max_trials,max_iterations);
  //initialize and open output files
  perf_filename = new char[CHAR_LEN];
  sprintf(perf_filename, "PerformanceAlg%i_M_%i_D_%.2f_S_%d_%s.txt",
	  alg_mode, change_mode,change_degree,change_speed,problem_instance);
  //for performance
  if ((log_performance = fopen(perf_filename,"a")) == NULL) { exit(2); }
  
  div_filename = new char[CHAR_LEN];
  sprintf(div_filename, "DiversityAlg%i_M_%i_D_%.2f_S_%d_%s.txt",
	  alg_mode, change_mode,change_degree,change_speed,problem_instance);
  //for diversity
  if ((log_diversity = fopen(div_filename,"a")) == NULL) { exit(2); }
  
  if(change_mode==4){  
    var_filename = new char[CHAR_LEN];
    sprintf(var_filename, "VaryingAlg%i_M_%i_S_Rand_D_Rand_%s.txt",
            alg_mode, change_mode, problem_instance);
    //for degree values
    if ((log_varying_values = fopen(var_filename,"a")) == NULL) { exit(2); }
  }
}

double mean(double* values, int size){   
  int i;
  double m = 0.0;  
  for (i = 0; i < size; i++){
      m += values[i];
  }
  m = m / (double)size;
  return m; //mean 
}

double stdev(double* values, int size, double average){
  int i;
  double dev = 0.0;
  
  if (size <= 1)
    return 0.0;
  for (i = 0; i < size; i++){
    dev += ((double)values[i] - average) * ((double)values[i] - average);
  }
  return sqrt(dev / (double)(size - 1)); //standard deviation
}

double calc_diversity(int size){
  int i, j;
   double div = 0.0; 
   for (i = 0; i < size; i++){
     for (j = 0; j < size; j++) {
       if (i != j) {
	 div += distance_between(&ant_population[i], &ant_population[j]);//common edges
       }
     }
   }  
   return (1.0 / (size * (size - 1))) * div; //population diversity
}

void close_stats(void){
  int i,j;
  double perf_mean_value, perf_stdev_value;
  double div_mean_value, div_stdev_value;
  double* perf_of_trials = new double[max_trials];
  double* perf_of_iterations = new double[max_iterations];
  double* div_of_trials = new double[max_trials];
  double* div_of_iterations = new double[max_iterations];
  //For graph plots
  for(i = 0; i < max_iterations; i++){
    for(j = 0; j < max_trials; j++){
      perf_of_iterations[i] += performance[j][i];
      div_of_iterations[i]+= diversity[j][i];
    }
    perf_of_iterations[i] /= ((double)max_trials);
    div_of_iterations[i]/= ((double)max_trials);
    fprintf(log_performance, "%.2f", perf_of_iterations[i]);
    fprintf(log_performance,"\n");
    fprintf(log_diversity, "%.2f", div_of_iterations[i]);
    fprintf(log_diversity,"\n");
  } 
  fprintf(log_performance,"\n");
  fprintf(log_diversity,"\n");
  fprintf(log_performance,"Statistical results\n");
  fprintf(log_diversity,"Statistical results\n");
  //For statistics
  for(i = 0; i < max_trials; i++){
    for(j =0; j < max_iterations; j++){
      perf_of_trials[i] += performance[i][j];
      div_of_trials[i] +=diversity[i][j];
    }
    perf_of_trials[i] /= ((double)max_iterations);
    div_of_trials[i]/=((double)max_iterations);
    fprintf(log_performance, "%.2f", perf_of_trials[i]);
    fprintf(log_performance,"\n");
    fprintf(log_diversity, "%.2f", div_of_trials[i]);
    fprintf(log_diversity,"\n");
  }  
  perf_mean_value = mean(perf_of_trials,max_trials);
  perf_stdev_value = stdev(perf_of_trials,max_trials,perf_mean_value);
  div_mean_value = mean(div_of_trials,max_trials);
  div_stdev_value = stdev(div_of_trials,max_trials,div_mean_value); 
  fprintf(log_performance,"Mean %f\t ",perf_mean_value);
  fprintf(log_performance,"\tStd Dev %f\t ",perf_stdev_value);
  fprintf(log_diversity,"Mean %f\t ",div_mean_value);
  fprintf(log_diversity,"\tStd Dev %f ",div_stdev_value);

  cout << "Mean " << perf_mean_value << " Std " << perf_stdev_value << endl; 
  
  fclose(log_performance);
  fclose(log_diversity);
  if(change_mode==4) fclose(log_varying_values);

  //free memory
  delete[] perf_of_trials;
  delete[] perf_of_iterations; 
  delete[] div_of_trials;
  delete[] div_of_iterations;
}

void statistics_and_output(int r, int t){
  //*NOTE*: change the best_so_far_ant and 
  //n_ants values according to your algorithm 
  //implementation
  performance[r-1][t-1] = best_so_far_ant->tour_length;
  diversity[r-1][t-1] = calc_diversity(n_ants);
  //output varying values when (change_mode=4)
  //the values for all RUNS are identical
  if(change_mode==4 && r==1){
    fprintf(log_varying_values, "%.2f", change_degree);
    fprintf(log_varying_values,"\n");
  }
  cout<<"iteration: " << current_iteration <<  " best_so_far " << best_so_far_ant->tour_length << endl;
}

void re_evaluate(){
  //*NOTE*: These actions stand only for ACO 
  //algorithms. In case an EA is used, simply
  //evaluate the individuals in the population 
  //using fitness_evaluation()
  compute_nn_lists();
  init_heuristic_info();
  compute_total_info();  
  //Reset best solution after a change to
  //calculate the modified offline performance
  best_so_far_ant->tour_length = INFTY;
}

void init_try(int t){
  //initialize algorithmic structures 
  //for the new environment
  compute_nn_lists();             //reset nearest neighbour cities
  init_heuristic_info();          //initialize heuristic info
  trail0 = 1.0 / nn_tour();       
  init_pheromone_trails(trail0);  //initialize pheromone trails
  compute_total_info();           //combine heuristic+pheromone
  best_so_far_ant->tour_length = INFTY; //reset best solution found
  if(alg_mode!=1) previous_best_so_far_ant->tour_length = INFTY;
  if(alg_mode==4) init_memory_randomly(); //for MIACO
}

void exit_try(){
  //free memory
  for(int i = 0; i < problem_size; i++){
   delete[] distances[i];
   delete[] traffic_factors[i];
 }
 delete[] distances;
 delete[] traffic_factors;
 if(change_mode ==2 || change_mode==3) {
    for(int i =0; i < cyclic_states; i++){
      for(int j =0; j < problem_size; j++){
      delete[] cyclic_bases[i].base[j];
       }
     } 
     delete[] cyclic_bases;
  }

 if(alg_mode == 4){
    for(int i = 0; i < long_memory_size; i++)
     delete[] long_memory[i].tour;
  } 
}

void free_memory(){
  delete[] init_objects;
  //free memory from the ACO implementation
  delete[] best_so_far_ant->tour;
  for(int i =0; i < n_ants; i++){
    delete[] ant_population[i].tour;
    delete[] ant_population[i].visited;
  }
  delete[] ant_population; 
  if(alg_mode!=1){
    delete[] previous_best_so_far_ant->tour;
    for(int i =0; i < 2; i++) { 
      delete[] previous_best[i].tour;
      delete[] previous_best[i].visited;
    }
    delete[] previous_best;
  }
  for(int i =0; i <problem_size; i++){
    delete[] pheromone[i];
    delete[] heuristic[i];
    delete[] total[i];
    delete[] nn_list[i];
  }
  delete[] pheromone;
  delete[] heuristic; 
  delete[] total;
  delete[] nn_list;
  delete[] prob_of_selection; 
  delete[] short_memory;
  delete[] div_filename;
  delete[] perf_filename;
  delete[] var_filename;
  delete[] long_memory; 
  delete[] random_point; 
  for(int i = 0; i < max_trials; i++) {
    delete[] performance[i];
    delete[] diversity[i];
  }
  delete[] performance;
  delete[] diversity; 
}

/****************************************************************/
/*                Main Function (ACODTSP)                       */
/****************************************************************/
int main(int argc, char *argv[]) {
  // read in parameters
  problem_instance = argv[1];
  change_mode = atoi(argv[2]);
  change_degree = atof(argv[3]);
  change_speed = atoi(argv[4]); 

  max_iterations = 1000;
  max_trials = 30;
  
  read_problem(problem_instance);   //Read TSP from file
  check_input_parameters();         //validate the parameters  

  open_stats();
  
  //ACO algorithms initialization
  set_algorithm_parameters();
  allocate_ants();
  allocate_structures();
  
  for(int run = 1; run <= max_trials; run++){
    cout<< "-------------Run: " << run << "------------------" << endl; 
    srand(1);                       //static seed reset for the changes
    seed = time(NULL) + run;        //changing seed for the algorithms  
    initialize_environment();       //set DTSP for a new Run
    
    current_iteration=1;            //set iteration for a new Run
    init_try(run);                  //re-initialize algorithmic structures
     
    while(current_iteration <= max_iterations){//termination not met
      //ACO algorithm methods
      construct_solutions();     
      update_best();   
      pheromone_update();
      //output results to files
      statistics_and_output(run,current_iteration);      
      //environmental changes (integration of ACO with DTSP with traffic)  
      if(current_iteration%change_speed == 0){ //period of change  
        change_environment();
        re_evaluate();
      }
      current_iteration++;
    }  
   exit_try(); 
  }
  close_stats();
   
  free_memory();
  return 0;
}
