///////////////////////////////////////////////////////////////
//
//   Peaks operation section start
#include "MovingPeak.h"
#include "Global.h"

 int MovingPeak::F=4;
 
//**************************************
//		F		Evaluation Function

//		1		constant_basis_func()
//		2		five_peak_basis_func()
//		3		peak_function1()	
//		4		peak_function_cone()
//		5		peak_function_hilly()
//		6		peak_function_twin()
//**************************************

 double MovingPeak::vlength = 1.0; /* distance by which the peaks are moved, severity */
 double MovingPeak::height_severity=7.0; /* severity of height changes, larger numbers  mean larger severity */
 double MovingPeak::width_severity = 1.0; /* severity of width changes, larger numbers mean larger severity */

/* lambda determines whether there is a direction of the movement, or whether
   they are totally random. For lambda = 1.0 each move has the same direction,
   while for lambda = 0.0, each move has a random direction */
 double MovingPeak::lambda=0;
 int MovingPeak::number_of_peaks = 10; /* number of peaks in the landscape */
 int MovingPeak::use_basis_function=0; /* if set to 1, a static landscape (basis_function) is included in the fitness evaluation */
 int MovingPeak::calculate_average_error=1; /* saves computation time if not needed and set to 0 */
 int MovingPeak::calculate_offline_performance = 1; /* saves computation time if not needed and set to 0 */
 int MovingPeak::calculate_right_peak = 1; /* saves computation time if not needed and set to 0 */

/* minimum and maximum height of the peaks          */
/* height chosen randomly when standardheight = 0.0 */
 double MovingPeak::minheight = 30.0, MovingPeak::maxheight = 70.0, MovingPeak::standardheight = 50.0;
/* width chosen randomly when standardwidth = 0.0 */
 double MovingPeak::minwidth = 1.0 , MovingPeak::maxwidth = 12.0, MovingPeak::standardwidth = 0.0; 
//0.8 7.0
 int MovingPeak::recent_change = 1; /* indicates that a change has just ocurred */
 int MovingPeak::current_peak;      /* peak on which the current best individual is located */
 int MovingPeak::maximum_peak;       /* number of highest peak */
 double MovingPeak::current_maximum; /* fitness value of currently best individual */ 
 double MovingPeak::offline_performance = 0.0;
 double MovingPeak::offline_error = 0.0;
 double MovingPeak::avg_error=0;    /* average error so far */ 
 double MovingPeak::current_error=0;/* error of the currently best individual */
 double MovingPeak::global_max;     /* absolute maximum in the fitness landscape */
 int MovingPeak::evals = 0;         /* number of evaluations so far */
 double **MovingPeak::peak=0;         /* data structure to store peak data */
 double *MovingPeak::shift=0;
 double *MovingPeak::coordinates=0;
 int *MovingPeak::covered_peaks=0;    /* which peaks are covered by the population ? */
 double **MovingPeak::prev_movement=0;/* to store every peak's previous movement */


/* initialize all variables at the beginning of the program */
void MovingPeak::init_peaks()
{
  int i,j;
  double dummy;
  
  shift = new double[Global::num_dim];
 // (double *) calloc(Global::num_dim, sizeof(double));
  coordinates = new double[Global::num_dim];
  //(double *) calloc(Global::num_dim, sizeof(double));
  covered_peaks = new int[number_of_peaks];
  //(int *) calloc(number_of_peaks, sizeof(int));
  peak = new double*[number_of_peaks];
  // (double **) calloc(number_of_peaks, sizeof(double*));
  prev_movement = new double*[number_of_peaks];
  //(double **) calloc(number_of_peaks, sizeof(double*));
  for (i=0; i< number_of_peaks; i++){
    peak[i]= new double[Global::num_dim+2];
	//(double *) calloc(Global::num_dim+2, sizeof(double));
    prev_movement[i] =new double[Global::num_dim];
	// (double *) calloc(Global::num_dim, sizeof(double));
  }
  for (i=0; i< number_of_peaks; i++)
    for (j=0; j< Global::num_dim; j++){
      peak[i][j] = 100.0*Global::uniform.Next();
      prev_movement[i][j] = Global::uniform.Next()-0.5;
    }
  if (standardheight <= 0.0)
    for (i=0; i< number_of_peaks; i++)
      peak[i][Global::num_dim+1]= (maxheight-minheight)*Global::uniform.Next()+minheight;
  else
    for (i=0; i< number_of_peaks; i++)
      peak[i][Global::num_dim+1]= standardheight;
  if (standardwidth <= 0.0)
    for (i=0; i< number_of_peaks; i++)
      peak[i][Global::num_dim]= (maxwidth-minwidth)*Global::uniform.Next()+minwidth;
  else
    for (i=0; i< number_of_peaks; i++)
      peak[i][Global::num_dim]= standardwidth;
  if(calculate_average_error){
    global_max = -100000.0;
    for (i=0;i<number_of_peaks; i++){
      for (j=0; j<Global::num_dim; j++)
        coordinates[j]=peak[i][j];
      dummy = dummy_eval(coordinates);
      if (dummy>global_max)
        global_max = dummy;
    }
  }
}

/* free disc space at end of program */
void MovingPeak::free_peaks()
{
	int i;

	for (i=0; i< number_of_peaks; i++){
	delete [] peak[i];
	delete [] prev_movement[i];
	}
	delete [] peak;
	delete [] prev_movement;

	delete []  shift ;
	delete [] coordinates;
	delete []  covered_peaks ;
	peak=0;
	prev_movement=0;
	shift=0 ;
	coordinates=0;
	covered_peaks=0 ;
}

/* current_peak_calc determines the peak of the current best individual */
void MovingPeak::current_peak_calc (double *gen)
{
  int i;
  double maximum = -100000.0, dummy;

  current_peak = 0;
  maximum = function_selection(gen, 0);
  for(i=1; i<number_of_peaks; i++){
    dummy = function_selection(gen, i);
    if (dummy > maximum){
      maximum = dummy;
      current_peak = i;
    }
  }
}


/* evaluation function */
double MovingPeak::eval_movpeaks (double *gen)
{
  int i;
  double maximum = -100000.0, dummy;

  //if ((change_frequency > 0)&&(evals%change_frequency==0))
   // change_peaks();

  for(i=0; i<number_of_peaks; i++)
    {
    dummy = function_selection(gen, i);
    if (dummy > maximum)
      maximum = dummy;
    }

  if (use_basis_function) {
    
    dummy = function_selection(gen,-1);
    /* If value of basis function is higher return it */
    if (maximum < dummy)
      maximum = dummy;
  }
  return(maximum);
}

/* dummy evaluation function allows to evaluate without being counted */
double MovingPeak::dummy_eval (double *gen)
{
  int i;
  double maximum = -100000.0, dummy;

  for(i=0; i<number_of_peaks; i++)
    {
    dummy = function_selection(gen, i);
    if (dummy > maximum)
      maximum = dummy;
    }

  if (use_basis_function) {
    
    dummy = function_selection(gen,-1);
    /* If value of basis function is higher return it */
    if (maximum < dummy)
      maximum = dummy;
  }
  return(maximum);
}
void MovingPeak::offline_error_eval(CPosition &p)
{
  if (calculate_average_error){
    avg_error+=global_max - p.fitness;
  }
  if (calculate_offline_performance){
    if(recent_change||(p.fitness > current_maximum)){
      current_error = global_max - p.fitness;
      if (calculate_right_peak)
		current_peak_calc(p.x);
      current_maximum = p.fitness;
//	if(p.fitness>global_max) 
//			cout<<"error";
      recent_change = 0;
    }
    offline_performance += current_maximum;
    offline_error+= current_error;
  }
}
/* whenever this function is called, the peaks are changed */ 
void MovingPeak::change_peaks() 
{
  int i,j;
  double sum, sum2, offset, dummy;

  for(i=0; i<number_of_peaks; i++)
    {
      /* shift peak locations */
    sum = 0.0;
    for (j=0; j<Global::num_dim; j++){
      shift[j]=Global::uniform.Next()-0.5;
      sum += shift[j]*shift[j];
      }
    if(sum>0.0)                 
      sum = vlength/sqrt(sum);
    else                           /* only in case of rounding errors */
      sum = 0.0;
    sum2=0.0;
    for (j=0; j<Global::num_dim; j++){
      shift[j]=sum*(1.0-lambda)*shift[j]+lambda*prev_movement[i][j];
      sum2 += shift[j]*shift[j];
    }
    if(sum2>0.0)                 
      sum2 = vlength/sqrt(sum2);
    else                           /* only in case of rounding errors */
      sum2 = 0.0;
    for(j=0; j<Global::num_dim; j++){
      shift[j]*=sum2;
      prev_movement[i][j]= shift[j];
	  if ((peak[i][j]+prev_movement[i][j]) < Global::boundary.lower){
        peak[i][j] = 2.0*Global::boundary.lower-peak[i][j]-prev_movement[i][j];
        prev_movement[i][j]*=-1.0;
      }
	  else if ((peak[i][j]+prev_movement[i][j]) > Global::boundary.upper){
        peak[i][j]= 2.0*Global::boundary.upper-peak[i][j]-prev_movement[i][j];
        prev_movement[i][j]*=-1.0;
      }
      else
        peak[i][j] += prev_movement[i][j];    
      }
    /* change peak width */
    j = Global::num_dim;
	offset = movnrand()*width_severity;
    if ((peak[i][j]+offset) < minwidth)
      peak[i][j] = 2.0*minwidth-peak[i][j]-offset;
    else if ((peak[i][j]+offset) > maxwidth)
      peak[i][j]= 2.0*maxwidth-peak[i][j]-offset;
    else
      peak[i][j] += offset;
   /* change peak height */
    j++;
    offset = height_severity*movnrand();
    if ((peak[i][j]+offset) < minheight)
      peak[i][j] = 2.0*minheight-peak[i][j]-offset;
    else if ((peak[i][j]+offset) > maxheight)
      peak[i][j]= 2.0*maxheight-peak[i][j]-offset;
    else
      peak[i][j] += offset;    
  }
  if(calculate_average_error){
    global_max = -100000.0;
    for (i=0;i<number_of_peaks; i++){
      for (j=0; j<Global::num_dim; j++)
        coordinates[j]=peak[i][j];
      dummy = dummy_eval(coordinates);
      if (dummy>global_max){
        global_max = dummy;
        maximum_peak = i;
      }
    }
  }
  recent_change = 1;
}


/* Basis Functions */

/* This gives a constant value back to the eval-function that chooses the max of them */
double MovingPeak::constant_basis_func(double *gen)
{
  return 0.0;
}

double MovingPeak::five_peak_basis_func(double *gen)
{
  int i,j;
  double maximum = -100000.0, dummy;
  static double basis_peak [5][7]=
{ 
  {8.0,  64.0,  67.0,  55.0,   4.0, 0.1, 50.0},  
  {50.0,  13.0,  76.0,  15.0,   7.0, 0.1, 50.0}, 
  {9.0,  19.0,  27.0,  67.0, 24.0, 0.1, 50.0}, 
  {66.0,  87.0,  65.0,  19.0,  43.0, 0.1, 50.0}, 
  {76.0,  32.0,  43.0,  54.0,  65.0, 0.1, 50.0}
};
  for(i=0; i<5; i++)
    {
      dummy = (gen[0]-basis_peak[i][0])*(gen[0]-basis_peak[i][0]);
      for (j=1; j< Global::num_dim; j++)
	dummy += (gen[j]-basis_peak[i][j])*(gen[j]-basis_peak[i][j]); 
      dummy = basis_peak[i][Global::num_dim+1]-(basis_peak[i][Global::num_dim]*dummy);
      if (dummy > maximum)
        maximum = dummy;
    }
  return maximum;
}
  


/* Peak Functions */

/* sharp peaks */
double MovingPeak::peak_function1 (double *gen, int peak_number)
{
  int j;
  double dummy;

  dummy = (gen[0]-peak[peak_number][0])*(gen[0]-peak[peak_number][0]); 
  for (j=1; j< Global::num_dim; j++)
    dummy += (gen[j]-peak[peak_number][j])*(gen[j]-peak[peak_number][j]); 
  return peak[peak_number][Global::num_dim+1]/(1+(peak[peak_number][Global::num_dim])*dummy);
}

double MovingPeak::peak_function_cone (double *gen, int peak_number)
{
  int j;
  double dummy;

  dummy =  (gen[0]-peak[peak_number][0])*(gen[0]-peak[peak_number][0]); 
  for (j=1; j< Global::num_dim; j++)
    dummy += (gen[j]-peak[peak_number][j])*(gen[j]-peak[peak_number][j]);
	dummy =peak[peak_number][Global::num_dim+1]-(peak[peak_number][Global::num_dim]*sqrt(dummy));

  return dummy;
}

double MovingPeak::peak_function_hilly (double *gen, int peak_number)
{
  int j;
  double dummy;

  dummy =  (gen[0]-peak[peak_number][0])*(gen[0]-peak[peak_number][0]); 
  for (j=1; j< Global::num_dim; j++)
    dummy += (gen[j]-peak[peak_number][j])*(gen[j]-peak[peak_number][j]); 
  return peak[peak_number][Global::num_dim+1]-(peak[peak_number][Global::num_dim]*dummy)-0.01*sin(20.0*dummy);
}

double MovingPeak::peak_function_twin (double  *gen, int peak_number) /* two twin peaks moving together */
{
  int j;
  double maximum = -100000.0, dummy;
  static double twin_peak [7] = /* difference to first peak */
  {
    1.0,  1.0,  1.0,  1.0,   1.0, 0.0, 0.0,
  };
  
  dummy = pow(gen[0]-peak[peak_number][0],2);
  for (j=1; j< Global::num_dim; j++)
     dummy += pow(gen[j]-peak[peak_number][j],2); 
  dummy = peak[peak_number][Global::num_dim+1]-(peak[peak_number][Global::num_dim]*dummy);
  maximum = dummy;
  dummy = pow(gen[j]-(peak[peak_number][0]+twin_peak[0]),2);
  for (j=1; j< Global::num_dim; j++)
     dummy += pow(gen[j]-(peak[peak_number][j]+twin_peak[0]),2); 
  dummy = peak[peak_number][Global::num_dim+1]+twin_peak[Global::num_dim+1]-((peak[peak_number][Global::num_dim]+twin_peak[Global::num_dim])*dummy);
  if (dummy > maximum)
    maximum = dummy;
  
  return maximum;
}  
double MovingPeak::function_selection(double  *gen, int peak_number){
	double dummy;
	switch(F){
	case 1: {
			dummy=constant_basis_func(gen);
			break;
		 }
	case 2: {
			dummy=five_peak_basis_func(gen);
			break;
		 }
		case 3: {
			dummy=peak_function1(gen, peak_number);
			break;
		 }
	case 4: {
			dummy=peak_function_cone (gen, peak_number);
			break;
		 }
	case 5: {
			dummy=peak_function_hilly (gen, peak_number);
			break;
		 }
	case 6: {
			dummy=peak_function_twin (gen, peak_number);
			break;
		 }
	}
	return dummy;
}
/* The following procedures may be used to change the step size over time */


void MovingPeak::change_stepsize_random () /* assigns vlength a value from a normal distribution */
{ 
  vlength = movnrand();
}
  
void MovingPeak::change_stepsize_linear() /* sinusoidal change of the stepsize, */
{
  static int counter = 1;
  static double frequency = 3.14159/20.0;  /* returns to same value after 20 changes */

  vlength = 1+ sin((double)counter*frequency);
  counter ++;
}

double MovingPeak::get_avg_error() /* returns the average error of all evaluation calls so far */
{
  return (avg_error/(double)evals);
}

double MovingPeak::get_current_error() /* returns the error of the best individual evaluated since last change */
     /* To use this function, calculate_average_error and calculate_offline_performance must be set */
{
  return current_error;
}

double MovingPeak::get_offline_performance() /* returns offline performance */
{
  return (offline_performance/(double)evals);
}

double MovingPeak::get_offline_error() /* returns offline error */
{
  return (offline_error/(double)evals);
}


int MovingPeak::get_number_of_evals() /* returns the number of evaluations so far */
{
  return evals;
}

int MovingPeak::get_right_peak()  /* returns 1 if current best individual is on highest peak, 0 otherwise */
{
  if (current_peak == maximum_peak)
    return 1;
  else
    return 0;
}

//
//   Peaks operation section ends
///////////////////////////////////////////////////////////////////////
