/**************************************************************************************/
/* ACODBGP.c                    VRP version 1.0                                       */
/* This is an implementation of the dynamic benchmark generator for permutation       */
/* encoded problems (DBGP) of the following paper:                                    */
/*                                                                                    */
/*   M. Mavrovouniotis, S. Yang and X. Yao. A benchmark generator for dynamic         */
/*   permutation-encoded problems. Proceedings of the 12th International Conference on*/ 
/*   Parallel Problem Solving from Nature (PPSN XII), Part II, LNCS, vol. 7492, pp.   */
/*   508-517, Springer-Verlag, 2012.                                                  */
/*                                                                                    */
/* Compile:                                                                           */
/*   g++ -o ACODBGP ACODBGP.c                                                         */
/* Run:                                                                               */
/*   ./ACODBGP problem_instance change_mode change_degree change_speed                */
/*                                                                                    */
/*   e.g., ./ACODBGP F-n45-k4.vrp 1 0.25 100                                          */
/*                                                                                    */
/* Written by:                                                                        */
/*   Michalis Mavrovouniotis, De Montfort University, UK; July 2013                   */
/*                                                                                    */
/* If any query, please email Michalis Mavrovouniotis at mmavrovouniotis@dmu.ac.uk    */ 
/*                                                                                    */
/**************************************************************************************/

#include<iostream>
#include<stdio.h>
#include<stdlib.h>
#include<string>
#include<cstring>
#include<math.h>
#include<fstream>
#include<cmath>
#include<time.h>
#include<limits.h>

using namespace std;

#define EPSILON  0.000000000000000000000001
#define INFTY    INT_MAX
#define CHAR_LEN 100

struct object { 
  int id;
  double x;
  double y;
};

struct object *init_objects;     //Actual objects of the instance
struct object *masked_objects;   //Masked objects of the instance
int *init_demand;                //Actual customer demands
int *masked_demand;              //Masked customer demands
int **distances;                 //Distance matrix
int problem_size;                //Size of the instance (includes depot)
int problem_size_no_depot;       //Size of the instance (excludes depot)
char* problem_instance;          //Name of the instance

int max_capacity;                //capacity of vehicles
int depot;                       //depot id (usually 0)

int change_mode;                 //How environment changes, reappear or not
                                 //1:random, 2:reappear(cyclic), 3:reappear(randomly), 4:varying
double change_degree;            //Ratio of swapped objects
int change_speed;                //Peiord of changes in algorithmic iterations

int cyclic_states = 4;           //Base states for environments that reappaear
bool noise_flag = false;         //whether noise is added or not
double p_noise = 0.05;           //Noise probability for environments that reappear

int cyclic_base_count;          //Base states index for cyclical reappeared environments
int cyclic_random;              //Base states index for randomly reappeared environments
int* random_vector;             //Mask of random objects
int* re_random_vector;          //Mask of re-ordered random objects
int** cyclic_random_vector;     //Masks of objects used in reappered environments
int** cyclic_re_random_vector;  //Masks of re-ordered objects used in reappeared environments
int* noise;                     //Mask of random objects for noisy environments
int* re_noise;                  //Mask of re-ordered objects for noisy environments

int current_iteration;          //Used to calculate the period of change
int seed;                       
int max_iterations;
int max_trials;


//Used to output offline performance
int** performance;
FILE *log_performance;
FILE *log_varying_values;

//output files
char * perf_filename; 
char * var_filename;

/****************************************************************/
/*Random number generator where the seed is the same in all runs*/
/****************************************************************/
double env_random_number(double low, double high) {
  return ((double)(rand()%10000)/10000.0)*(high - low) + low;  
}

/****************************************************************/
/*Generate a vector of random objects used for the swaps        */
/****************************************************************/
int * generate_random_vector(int size){
  int i, help, node, tot_assigned = 0;
  double rnd;
  int  *r;
  r = new int[size];
  //input object indexes
  for ( i = 0 ; i < size; i++){
    r[i] = i+1; //exclude depot
  } 
  for ( i = 0 ; i < size ; i++ ) {
    //find an index for a object randomly 
    rnd  = env_random_number(0.0,1.0);
    node = (int) (rnd  * (size - tot_assigned));//random number 
    help = r[i];
    r[i] = r[i+node];
    r[i+node] = help;
    tot_assigned++;
  }
  return r;
}

/****************************************************************/
/*Re-order the first random objects vector to swap objects      */
/*according to the degree of change                             */
/****************************************************************/
int * generate_reordered_random_vector(){
  int i, changes, help, r_index;
  int *r;  
  //set swaps on the first "changes" objects
  changes = (int)abs(change_degree * (problem_size_no_depot)); 
  r = new int[changes];
  //input object indexes
  for (i = 0; i < changes; i++){
    r[i] = random_vector[i];
  }
  //re-order the random_vector randomly
  for (i = 0; i < changes; i++){
    help = r[i];   
    r_index = (int)(env_random_number(0.0,1.0) * (double)changes);//random number 0..changes
    r[i] = r[r_index];
    r[r_index] = help;
  }
  return r;
}

/****************************************************************/
/*Compute and return the euclidean distance of two objects      */
/****************************************************************/
int euclidean_distance(int i, int j) {
  double xd,yd;
  int r = 0.0;
  xd = masked_objects[i].x - masked_objects[j].x;
  yd = masked_objects[i].y - masked_objects[j].y;
  r  = sqrt(xd*xd + yd*yd) + 0.5; 
  return r;
}

/****************************************************************/
/*Compute the distance matrix of the problem instance           */
/****************************************************************/
void compute_distances(void) {
  for(int i = 0; i <problem_size; i++){
    for(int j = 0; j < problem_size; j++){
      distances[i][j] = euclidean_distance(i,j);    
    }
  }
}

/****************************************************************/
/*Swap the location between two objects of the problem instance */
/****************************************************************/
void swap_masked_objects(int obj1, int obj2){
  double help1, help2; 
  int help3; 
  help1=help2 = 0.0;
  help3=0; 
  help1 = masked_objects[obj1].x;
  help2 = masked_objects[obj1].y; 
  masked_objects[obj1].x = masked_objects[obj2].x;
  masked_objects[obj1].y = masked_objects[obj2].y;  
  masked_objects[obj2].x = help1;
  masked_objects[obj2].y = help2;

  help3 = masked_demand[obj1];
  masked_demand[obj1] = masked_demand[obj2];
  masked_demand[obj2] = help3;
}

/****************************************************************/
/* Perform random changes by swapping the first                 */
/* (change_degree * problem_size) objects randomly              */
/****************************************************************/
void add_random_change(void){
  int i, changes,  obj1, obj2; 
  changes = (int)abs(change_degree * (problem_size_no_depot)); //exclude depot
  random_vector = generate_random_vector(problem_size_no_depot);
  re_random_vector = generate_reordered_random_vector(); 
  //swap locations of objects
  for (i = 0; i < changes; i++){
    obj1 = masked_objects[random_vector[i]].id;
    obj2 = masked_objects[re_random_vector[i]].id;
    swap_masked_objects(obj1, obj2);  
  }
}

/****************************************************************/
/* Add noise to a small portion of objects. This method is used */
/* for cyclic environments with noise.                          */
/****************************************************************/
void add_noise(){
  int i, noise_change, obj1, obj2;
  noise_change = (int)abs(p_noise * (problem_size_no_depot)); //exclude depot
  noise = generate_random_vector(problem_size_no_depot);
  re_noise = generate_random_vector(problem_size_no_depot);
  //swap locations of objects
  for (i = 0; i < noise_change; i++){
    obj1 = masked_objects[noise[i]].id;
    obj2 = masked_objects[re_noise[i]].id;
    swap_masked_objects(obj1, obj2);
  }
}

/****************************************************************/
/* Remove the previously added noise. This method is used       */
/* for cyclic environments with noise.                          */
/****************************************************************/
void remove_noise(){
  int i, noise_change, obj1, obj2;
  noise_change = (int)abs(p_noise * (problem_size_no_depot)); //exclude depot
  //reverse the noisy objects
  for (i = noise_change - 1; i >= 0; i--){
    obj1 = masked_objects[re_noise[i]].id;
    obj2 = masked_objects[noise[i]].id;
    swap_masked_objects(obj1, obj2);
  }

  //free memory
  delete[] noise;
  delete[] re_noise;
}

/****************************************************************/
/* Remove the previously object swaps and returns the problem   */
/* instance to its initial state. Hence, the dynamic changes    */ 
/* are not incremental                                          */
/****************************************************************/
void reverse_changes(){
   int i, obj1, obj2; 
  //remove noise first if any
  if (noise_flag == true) remove_noise();
  //reverse swapped objects         
  int changes = (int)abs(change_degree * (problem_size_no_depot)); //exclude depot
  for (i = changes - 1; i >= 0; i--){
    obj1 = masked_objects[re_random_vector[i]].id;
    obj2 = masked_objects[random_vector[i]].id;    
    swap_masked_objects(obj1, obj2);
  }

  //free memory
  if(change_mode==1){
    delete[] random_vector;
    delete[] re_random_vector;
  }
}

/****************************************************************/
/*Generate and return a two-dimension array of type int         */
/****************************************************************/
int ** generate_2D_matrix_int(int n, int m){
  int **matrix;
  matrix = new int*[n];
  for ( int i = 0 ; i < n ; i++ ) {
    matrix[i] = new int[m];
  }
  //initialize 2-d array
  for(int i = 0; i < n; i++){
    for(int j = 0; j < m; j++) {
      matrix[i][j] = 0;
    }
  }
  return matrix;
}

/****************************************************************/
/* Construct the base state for the environments that reappear  */ 
/* Note that this method uses the cyclic_states global variable */    
/* to generate and store a number of "cyclic_states" vectors    */
/****************************************************************/
void generate_cyclic_environment(){
  int i, j, changes;
  changes = (int)abs(change_degree * (problem_size_no_depot));
  cyclic_random_vector = generate_2D_matrix_int(cyclic_states,changes);
  cyclic_re_random_vector = generate_2D_matrix_int(cyclic_states,changes);   
  //generate the base states according to the cyclic states
  for (i = 0; i < cyclic_states; i++){
    random_vector = generate_random_vector(problem_size_no_depot);
    re_random_vector = generate_reordered_random_vector();                
    for (j = 0; j < changes; j++){
      cyclic_random_vector[i][j] = random_vector[j];
      cyclic_re_random_vector[i][j] = re_random_vector[j];
    }
  }  
}

/****************************************************************/
/* Perfom an indexed, using the state variable, change based on */
/* the vectors generated in the generate_cyclic_environment     */ 
/****************************************************************/
void add_cyclic_change(int state){
  int i, changes, obj1, obj2;
  changes = (int)abs(change_degree * (problem_size_no_depot));
  //swap locations of objects from "state" index
  for (i = 0; i < changes; i++){
    obj1 = masked_objects[cyclic_random_vector[state][i]].id;
    obj2 = masked_objects[cyclic_re_random_vector[state][i]].id;
    random_vector[i] = cyclic_random_vector[state][i];
    re_random_vector[i] = cyclic_re_random_vector[state][i];   
    swap_masked_objects(obj1, obj2);              
  }
  //add noise if any 
  if (noise_flag == true) add_noise();  
}

/****************************************************************/
/* Perfom a dynamic change with randomly generated change_speed */
/* and random change_degree.                                    */
/****************************************************************/
void varying_environmental_changes(){
  int i, changes, obj1, obj2;
  change_speed = (int)env_random_number(1,100);//random number 1..100
  change_degree = env_random_number(0.0,1.0);  //random number 0..1
  changes = (int)abs(change_degree * (problem_size_no_depot));
  random_vector = generate_random_vector(problem_size_no_depot);
  re_random_vector = generate_reordered_random_vector(); 
  //swap locations of objects
  for (i = 0; i < changes; i++){
    obj1 = masked_objects[random_vector[i]].id;
    obj2 = masked_objects[re_random_vector[i]].id;
    swap_masked_objects(obj1, obj2);
  }

  //free memory
  delete[] random_vector;
  delete[] re_random_vector;
}

/****************************************************************/
/* Read the problem instance and generate the initial object    */
/* vector.                                                      */
/****************************************************************/
void read_problem(char* filename){
  char line[CHAR_LEN];
  char * keywords;
  char Delimiters[] = " :=\n\t\r\f\v";
  ifstream fin(filename);
  while((fin.getline(line, CHAR_LEN-1))){
    if(!(keywords = strtok(line, Delimiters)))
      continue;
    if(!strcmp(keywords, "DIMENSION")){			
      if(!sscanf(strtok(NULL, Delimiters), "%d", &problem_size)){
	cout<<"DIMENSION error"<<endl;
	exit(0);
      }
    }
    else if(!strcmp(keywords, "EDGE_WEIGHT_TYPE")){
      char * tempChar;
      if(!(tempChar=strtok(NULL, Delimiters))){
	cout<<"EDGE_WEIGHT_TYPE error"<<endl;
	exit(0);
      }
      if(strcmp(tempChar, "EUC_2D")){
	cout<<"not EUC_2D"<<endl;
	exit(0);
      }
    }
    else if (!strcmp(keywords, "CAPACITY")){
       if(!sscanf(strtok(NULL,Delimiters), "%d", & max_capacity)){
          cout << "CAPACITY error" << endl;
          exit(0); 
       }
    }
    else if(!strcmp(keywords, "NODE_COORD_SECTION")){
      if(problem_size!=0){ 
      problem_size_no_depot = problem_size - 1;
      masked_objects = new object[problem_size];
      init_objects = new object[problem_size];
      int i;
      for(i=0; i<problem_size; i++){
	//store initial objects
        fin>>init_objects[i].id;       
	fin>>init_objects[i].x>>init_objects[i].y;
        init_objects[i].id -=1;
	//initialize masked_objects with initial objects
        masked_objects[i].x=init_objects[i].x;
        masked_objects[i].y=init_objects[i].y;
        masked_objects[i].id=init_objects[i].id;
      }
      //compute the distances using initial objects
       distances = generate_2D_matrix_int(problem_size, problem_size);
       compute_distances();
     }
    }
    else if(!strcmp(keywords, "DEMAND_SECTION")){
     if(problem_size!=0){
       int temp;
       masked_demand = new int[problem_size];
       init_demand = new int[problem_size];
       for(int i =0; i < problem_size; i++){
        fin >> temp;
        fin >> init_demand[temp-1];
        masked_demand[i] = init_demand[i];
       }
     }
    else if(!strcmp(keywords, "DEPOT_SECTION")){        
      fin >> depot;  
      depot-=1;  
     }
    }
  }
  fin.close();
}

/****************************************************************/
/* Initialize the environment with the initial objects and      */ 
/* perform the initial dynamic change/generate base states      */
/****************************************************************/
void initialize_environment(){
  //initialize masked_objects with initial objects
  for(int i = 0; i < problem_size; i++) {
   masked_objects[i].x = init_objects[i].x;
   masked_objects[i].y = init_objects[i].y;
   masked_objects[i].id = init_objects[i].id;
   masked_demand[i] = init_demand[i]; 
  }
  
  switch(change_mode) {
  case 1: //random environment
    add_random_change();
    break;
  case 2:
  case 3://reappear environments
    generate_cyclic_environment();
    cyclic_base_count = 0;
    add_cyclic_change(cyclic_base_count);
    cyclic_random = 0;
    break;
  case 4: //varying environments
    change_speed = 10; //reset
    change_degree = 0.1;
    varying_environmental_changes();
    break;
  }
  compute_distances(); 
}

/****************************************************************/
/* Perform the dynamic change every "period" iteration, i.e.,   */ 
/* swapping objects                                             */
/****************************************************************/
void change_environment(int period){
   //period of change
   switch(change_mode){ 
    case 1: // random
      reverse_changes();
      add_random_change(); 
     break;
    case 2: //reappear cyclic pattern
      //cycle on the base states as a fixed logical ring
      if(cyclic_base_count == cyclic_states-1)
	cyclic_base_count = 0;
      else
	cyclic_base_count++;
      reverse_changes();  
      add_cyclic_change(cyclic_base_count);
      break;
    case 3: //reappear random pattern  
      cyclic_base_count = rand() % cyclic_states;
      //avoid the same base state
      while(cyclic_random == cyclic_base_count) cyclic_base_count = rand() % cyclic_states;
      cyclic_random = cyclic_base_count;
      reverse_changes();
      add_cyclic_change(cyclic_base_count);  
      break;
    case 4: // varying
      varying_environmental_changes();
      break;          
    }  
    compute_distances();
}

/****************************************************************/
/* Validate the values input by the user                        */
/****************************************************************/
void check_input_parameters() {

 if(problem_size == 0) {
   cout << "wrong problem instance file" << endl;
   exit(2);
 }
 if(change_mode > 4 || change_mode <= 0) {
   cout << "select a valid change mode (between 1..4)" << endl;
   exit(2);
 }
 if(change_degree > 1.0 || change_degree <= 0.0) {
   cout << "select a valid change degree (between 0..1)" << endl;
   exit(2);
 }
 if(change_speed <= 0) {
   cout << "select a valid change speed (int)" << endl;
   exit(2);
 }
}

/****************************************************************/
/* Evaluate a TSP tour and return the length. This method can   */
/* be used in the optimizer integrated with DBGP. The TSP tour  */
/* to input is an array of integers, where each integer         */
/* corresponds to an object                                     */
/****************************************************************/
int fitness_evaluation(int *t, int *id) {
  
  int i, one, two;
  int tour_length = 0;
  tour_length = distances[0][t[0]]; //depot - first customer
  for (i = 0; i < problem_size_no_depot-1; i++) {
    if(id[i] == id[i+1]){
       tour_length += distances[t[i]][t[i+1]];
    }else{
      tour_length += distances[t[i]][0];
      tour_length += distances[0][t[i+1]];
    }
  }
  tour_length += distances[t[problem_size_no_depot-1]][0]; //last customer - depot
  
  return tour_length;
}


//end DBGP


/**************************************************************************************/      
/*                                                                                    */
/*  ACO methods are based on the original ACO framework implementation:               */ 
/*                                                                                    */
/*    Thomas Stuetzle. ACOTSP, Version 1.02. Available from                           */
/*    http://www.aco-metaheuristic.org/aco-code, 2004.                                */           
/*                                                                                    */
/* Written by:                                                                        */
/*   Michalis Mavrovouniotis, De Montfort University, UK; refined July 2013           */
/*                                                                                    */
/* If any query, please email Michalis Mavrovouniotis at mmavrovouniotis@dmu.ac.uk    */ 
/*                                                                                    */
/**************************************************************************************/

#define IA       16807
#define IM       2147483647
#define AM       (1.0/IM)
#define IQ       127773
#define IR       2836
#define MASK     123459876

//Ant or individual structure that represent the TSP tour and tour cost
struct ant{
  int *tour;
  bool *visited;
  int *routes;      //shows the vehicles that cover the customers
  int tour_length;
  int capacity;     //vehicles current capacity
  int dummy_depots; //No. of vehicles
  bool start_new;   //start new route
  int no_customers; //No. of customers
};

int alg_mode = 1; //1: Ant System

struct ant *ant_population;   //Population of ants or individuals
ant *best_so_far_ant;         //current best so far tour 

double **pheromone;          //Pheromone matrix
double **heuristic;          //Heuristic information matrix
double **total;              //Pheromone + Heuristic information matrix

double *prob_of_selection;   //Selection probabilities

//General ACO parameters 
double alpha;
double beta;
double q_0;
double trail0;
double rho;

int n_ants;                //Population size
int depth;                 //Candidate list size (nearest neighbour)
int **nn_list;             //Candidate lists

/****************************************************************/
/*                     Initialization                           */
/****************************************************************/
void set_algorithm_parameters(void){
  //change the algorithm parameters according to your needs
  alpha = 1;
  beta = 5;
  q_0 = 0.0;
  rho = 0.6;
  n_ants = 50;
  depth = 20;

}

void allocate_ants(void){
  int i;  

  ant_population = new ant[n_ants];
  for(i = 0; i < n_ants; i++){
    ant_population[i].tour = new int[problem_size_no_depot];
    ant_population[i].visited = new bool[problem_size_no_depot];
    ant_population[i].routes = new int[problem_size_no_depot];
  }

  best_so_far_ant = new ant;
  best_so_far_ant->tour = new int[problem_size_no_depot];
  best_so_far_ant->visited = new bool[problem_size_no_depot];
  best_so_far_ant->routes = new int[problem_size_no_depot];

  prob_of_selection = new double[depth +1];
} 

double ** generate_2D_matrix_double(int n, int m){
  double **matrix;

  matrix = new double*[n];
  for ( int i = 0 ; i < n ; i++ ) {
    matrix[i] = new double[m];
  }
  //initialize the 2-d array
  for(int i = 0; i < n; i++){
    for(int j = 0; j < m; j++) {
      matrix[i][j] = 0.0;
    }
  }
  return matrix;
}


void allocate_structures(void){
  int i;

  pheromone = generate_2D_matrix_double(problem_size,problem_size);
  heuristic = generate_2D_matrix_double(problem_size,problem_size);
  total = generate_2D_matrix_double(problem_size,problem_size);
  nn_list = generate_2D_matrix_int(problem_size,depth);
}

void swap(int v[], int v2[],  int i, int j){
  int tmp;
  
  tmp = v[i];
  v[i] = v[j];
  v[j] = tmp;
  tmp = v2[i];
  v2[i] = v2[j];
  v2[j] = tmp;
}

void sort(int v[], int v2[], int left, int right){
  int k, last;

  if (left >= right) 
    return;
  swap(v, v2, left, (left + right)/2);
  last = left;
  for (k=left+1; k <= right; k++)
    if (v[k] < v[left])
      swap(v, v2, ++last, k);
  swap(v, v2, left, last);
  sort(v, v2, left, last);
  sort(v, v2, last+1, right);
}

void compute_nn_lists( void ){
  int i,j;
  int *distance_vector;
  int *help_vector;

  distance_vector = new int[problem_size];
  help_vector = new int[problem_size];
  //compute the nearest neigbhours of the objects
  for (j  = 0 ; j < problem_size; j++ ) { 
    for (i = 0 ; i < problem_size; i++ ) { 
      distance_vector[i] = distances[j][i];
      help_vector[i] = i;
    }
    distance_vector[j] = INT_MAX;
    sort(distance_vector, help_vector, 0, problem_size-1);
    for (i = 0 ; i < depth ; i++ ) {
      nn_list[j][i] = help_vector[i];
    }
  }

  //free memory
  delete[] distance_vector;
  delete[] help_vector;
}

void init_pheromone_trails(double initial_trail){
  int i,j;
  
  for(i = 0; i < problem_size; i++){
    for(j = 0; j <= i; j++){
      pheromone[i][j] = initial_trail;
      pheromone[j][i] = initial_trail;
    }
  } 
}

void init_heuristic_info(void){
  int i,j;
  
  for(i = 0; i < problem_size; i++){
    for(j = 0; j <= i; j++){
      heuristic[i][j] = 1.0/(double)(distances[i][j] + EPSILON); //small value to avoid 1 div 0
      heuristic[j][i] = heuristic[i][j];
    }
  }
}

void compute_total_info(void){
  int i,j;
  
  for(i = 0; i < problem_size; i++){
    for(j = 0; j < i; j++){
      total[i][j] = pow(pheromone[i][j],alpha) * pow(heuristic[i][j],beta);
      total[j][i] = total[i][j];
    }
  }
}

/****************************************************************/
/*                    Construct Solutions                       */
/****************************************************************/
void ant_empty_memory(ant *a){
  int i;
  //clear previous ant solution
  for(i = 0; i < problem_size; i++){
    a->visited[i] = false;
    a->routes[i] = 0;
  } 
  a->capacity = max_capacity;
  a->dummy_depots = 0;
  a->start_new = false;
  a->no_customers=0;
 
}

double alg_random_number(int *idum){
  int k;
  double ans;
  //uniformly distributed random number [0,1]
  k =(*idum)/IQ;
  *idum = IA * (*idum - k * IQ) - IR * k;
  if (*idum < 0 ) *idum += IM;
  ans = AM * (*idum);
  return ans;
}

void place_ant(ant *a){
  a->start_new = true;
  //place ants to the depot
  a->visited[depot] = true;
  a->dummy_depots+=1; //start first vehicle route     
}

void update_capacity(ant *a, int c, int p){
  int cap; 
  cap = a->capacity - masked_demand[c]; // current capacity
  if(cap >= 0) { //add customer 
    a->capacity -= masked_demand[c];
    a->visited[c] = true;
    a->tour[p] = c;
    a->routes[p] = a->dummy_depots;
    a->no_customers+=1;
  }else{ //start new route 
    a->dummy_depots+=1;
    a->capacity = max_capacity;
    a->start_new = true; 
  }
}

void choose_best_next(ant *a, int phase){
  int i,current, next, value_best;
  value_best=0;
  next = problem_size_no_depot-1;
  
  if( a->start_new == true){
    current = depot; //depot
    a->visited[depot] = true;
    a->start_new = false;
  }else{ //current object of ant
    current = a->tour[phase-1];
  } 
  value_best = -1.0;  //values in the list are always >=0.0
  //choose the next object with maximal (pheromone+heuristic) value 
  //among all objects of the problem 
  for(i = 0; i < problem_size; i++){
    if(a->visited[i] == false){
      //if object not visited
      if(total[current][i] > value_best){
	next = i;
	value_best = total[current][i];
      }
    }
  }
  update_capacity(a,next,phase);
}

void neighbour_choose_best_next(ant *a, int phase){
  int i,current,next,temp, value_best, help;
  next = problem_size_no_depot-1;
  if( a->start_new == true){
    current = depot; //depot
    a->visited[depot] = true;
    a->start_new = false;
  }else{ //current object of ant
    current = a->tour[phase-1];
  }
  value_best = -1.0; //values in the list are always >=0.0
  //choose the next object with maximal (pheromone+heuristic) value 
  //among all the nearest neighbour objects
  for(i =0; i < depth; i++){
    temp = nn_list[current][i];
    if(a->visited[temp] == false){
      //if object not visited
      help = total[current][temp];
      if(help > value_best){
	value_best=help;
	next = temp;
      }
    }
  }
  if(next == problem_size_no_depot-1){ 
    //if all nearest neighnour objects are already visited
    choose_best_next(a,phase);
  }else {
    update_capacity(a,next,phase);    
  }
}

void neighbour_choose_and_move_to_next(ant *a, int phase){
  int i,help, current, select;
  double rnd;
  double partial_sum = 0.0;
  double sum_prob = 0.0;
  double *prob_ptr;
  
  if((q_0 > 0.0) && (alg_random_number(&seed) < q_0)) {
    //with probability q_0 make the best possible choice
    neighbour_choose_best_next(a,phase);
    return;
  }    
  prob_ptr = prob_of_selection; //selection probabilities of the nearest neigbhour objects
  if( a->start_new == true){
    current = depot; //depot
    a->visited[depot] = true;
    a->start_new = false;
  }else{
    current = a->tour[phase-1];
  }
  //compute selection probabilities of nearest neigbhour objects
  for(i = 0; i < depth; i++){
    if(a->visited[nn_list[current][i]] ){
      prob_ptr[i] = 0.0;
    }else{
      prob_ptr[i] = total[current][nn_list[current][i]];
      sum_prob +=prob_ptr[i];
    }
  } 
  if(sum_prob <=0.0){
    //in case all neigbhbour objects are visited
    choose_best_next(a,phase);
  } else{
    //proabilistic selection (roullete wheel) 
    rnd = alg_random_number(&seed);
    rnd *= sum_prob;
    select = 0;
    partial_sum = prob_ptr[select];
    while(partial_sum<=rnd){
      select++;
      partial_sum+=prob_ptr[select];       
    }
    //this may very rarely happen because of rounding if 
    //rnd is close to 1
    if(select==depth){
      neighbour_choose_best_next(a,phase);     
      return;
    }
    help = nn_list[current][select];  
    update_capacity(a,help,phase);
  }
}

void construct_solutions(void){
  int k,all_ants;
  //clear memory of ants
  for(k =0; k < n_ants; k++){
    ant_empty_memory(&ant_population[k]);
  } 
  //place ants on the depot
  for(k =0; k < n_ants; k++){
    place_ant(&ant_population[k]); 
  }
  all_ants =0;
  //select object until all objects are visited
  while(all_ants < n_ants){ 
    for(k =0; k < n_ants; k++){
      if(ant_population[k].no_customers < problem_size_no_depot)
	neighbour_choose_and_move_to_next(&ant_population[k],ant_population[k].no_customers);      
      else
	all_ants++; //until all ants satisfied all customer demands
    }  
  }
  for(k =0; k < n_ants; k++){
    ant_population[k].tour_length = fitness_evaluation(ant_population[k].tour,ant_population[k].routes);//evalute 
  }
}

void choose_closest_next(ant *a, int phase){
  int i,current,next,min;  
  next = problem_size_no_depot-1;
  if( a->start_new == true){
    current = depot; //depot
    a->visited[depot] = true;
    a->start_new = false; 
  }else{
    current = a->tour[phase-1];
  }
  min = INFTY;
  //choose closest object used in the nn_tour()
  for(i = 0; i < problem_size; i++){
    if(a->visited[i] == false){
      //if object not visited
      if(distances[current][i] < min){
	next = i;
	min = distances[current][i];
      }
    } 
  }
  update_capacity(a,next,phase);
}

int nn_tour(void){
  int help=0; 
  ant_empty_memory(&ant_population[0]);
  place_ant(&ant_population[0]);
  //compute the tour length of the nearest neigbour heuristic
  //used to initialize the pheromone trails
  while(ant_population[0].no_customers < problem_size_no_depot){ 
    choose_closest_next(&ant_population[0],ant_population[0].no_customers);
  }
  ant_population[0].tour_length = fitness_evaluation(ant_population[0].tour,ant_population[0].routes);
  help = ant_population[0].tour_length;
  ant_empty_memory(&ant_population[0]);
  return help;
}

/****************************************************************/
/*                    Pheromone Update                          */
/****************************************************************/
void global_pheromone_deposit(ant *a){
  int i,j,h;
  double d_tau;
  d_tau = 1.0 / (double) a->tour_length;
  
  pheromone[0][a->tour[0]]+=d_tau;
  pheromone[a->tour[0]][0]= pheromone[0][a->tour[0]];
  for(i = 0; i < problem_size_no_depot-1; i++){
    j = a->tour[i];
    h = a->tour[i+1];
    if(a->routes[i] == a->routes[i+1]){
      pheromone[j][h]+= d_tau;
      pheromone[h][j] = pheromone[j][h];      
    } else {
      pheromone[j][0]+= d_tau;
      pheromone[0][j] = pheromone[j][0];
      pheromone[0][h]+= d_tau;
      pheromone[h][0] = pheromone[0][h];
    }
  }
  pheromone[a->tour[problem_size_no_depot-1]][0] += d_tau;
  pheromone[0][a->tour[problem_size_no_depot-1]] = pheromone[a->tour[problem_size_no_depot-1]][0];
}

void evaporation(void){
 int i, j;

 for (i = 0 ; i < problem_size; i++) {
	for (j = 0 ; j <= i; j++) {
	    pheromone[i][j] = (1 - rho) * pheromone[i][j];
	    pheromone[j][i] = pheromone[i][j];
	}
    }
}

void as_pheromone_update(void){
  int k;

  evaporation();
  for(k = 0; k < n_ants; k++){
     global_pheromone_deposit(&ant_population[k]);
  }
}


void pheromone_update(){   
  if(alg_mode == 1){
     
    as_pheromone_update();
  }  
  //other ACO algorithms can be added here since most 
  //of them differ on the way pheromone is updated
  compute_total_info(); //heuristic info + pheromone trails
}

/****************************************************************/
/*                    Update Best Ants                          */
/****************************************************************/
int find_best(void){
  int k,k_min, min;
  min = ant_population[0].tour_length;
  k_min = 0; 
  for(k = 1; k < n_ants; k++){
    if(ant_population[k].tour_length < min) {
      min=ant_population[k].tour_length;
      k_min = k;
    }
  }
  return k_min; //population best ant index
}

void copy_from_to(ant *a1, ant *a2){
  int i; 
  //ant2 is a copy of ant1
  a2->tour_length = a1->tour_length;
  a2->dummy_depots = a1->dummy_depots;
  for(i = 0; i < problem_size_no_depot; i++){
    a2->tour[i] = a1->tour[i];
    a2->routes[i] = a1->routes[i];
  }
}


void update_best(void){
  int iteration_best = find_best(); 
  
  if(ant_population[iteration_best].tour_length < best_so_far_ant->tour_length) {
    copy_from_to(&ant_population[iteration_best],best_so_far_ant);
  }
}

/****************************************************************/
/*                Update Statistics and Output                  */
/****************************************************************/
void open_stats(void){
  performance = generate_2D_matrix_int(max_trials,max_iterations);
  //initialize and open output files
  perf_filename = new char[CHAR_LEN];
  sprintf(perf_filename, "PerformanceAlg%i_M_%i_D_%.2f_S_%d_%s.txt",
	  alg_mode, change_mode,change_degree,change_speed,problem_instance);
  //for performance
  if ((log_performance = fopen(perf_filename,"a")) == NULL) { exit(2); }  
  if(change_mode==4){  
    var_filename = new char[CHAR_LEN];
    sprintf(var_filename, "VaryingAlg%i_M_%i_S_Rand_D_Rand_%s.txt",
            alg_mode, change_mode, problem_instance);
    //for degree values
    if ((log_varying_values = fopen(var_filename,"a")) == NULL) { exit(2); }
  }
}

double mean(double* values, int size){   
  int i;
  double m = 0.0;  
  for (i = 0; i < size; i++){
      m += values[i];
  }
  m = m / (double)size;
  return m; //mean 
}

double stdev(double* values, int size, double average){
  int i;
  double dev = 0.0;
  
  if (size <= 1)
    return 0.0;
  for (i = 0; i < size; i++){
    dev += ((double)values[i] - average) * ((double)values[i] - average);
  }
  return sqrt(dev / (double)(size - 1)); //standard deviation
}

double distance_between( ant *a1, ant *a2){
  int i, j, h, pos, pred;
  int distance;
  int *pos2 = new int[problem_size_no_depot];   
  //indexes of cities of ant2
  for (int i = 0 ; i < problem_size_no_depot; i++ ) {
    pos2[a2->tour[i]] = i;
  }  
  distance = 0;
  for (i = 0 ; i < problem_size_no_depot-1; i++ ) {
    if(a1->routes[i] == a2->routes[i+1]){
      j = a1->tour[i];
      h = a1->tour[i+1];
      pos = pos2[j];
      if (a2->tour[pos+1] == h)
	distance++;  //common edges
    }
  }  
  //free memory
  delete[] pos2; 

  return 1.0 - (distance/(double)(problem_size_no_depot + ((a1->dummy_depots+a2->dummy_depots) / 2.0)));
}

void close_stats(void){
  int i,j;
  double perf_mean_value, perf_stdev_value;
  double* perf_of_trials = new double[max_trials];
  double* perf_of_iterations = new double[max_iterations];
  //Initialize
  for(int i =0; i < max_iterations; i++) {
   perf_of_iterations[i] = 0.0;
   }
  for(int i =0; i < max_trials; i++){
     perf_of_trials[i] = 0.0;
   }
  //For graph plots
  for(i = 0; i < max_iterations; i++){
    for(j = 0; j < max_trials; j++){
      perf_of_iterations[i] += performance[j][i];
    }
    perf_of_iterations[i] /= ((double)max_trials);
    fprintf(log_performance, "%.2f", perf_of_iterations[i]);
    fprintf(log_performance,"\n");
  } 
  fprintf(log_performance,"\n");
  fprintf(log_performance,"Statistical results\n");
  //For statistics
  for(i = 0; i < max_trials; i++){
    for(j =0; j < max_iterations; j++){
      perf_of_trials[i] += performance[i][j];
    }
    perf_of_trials[i] /= ((double)max_iterations);
    fprintf(log_performance, "%.2f", perf_of_trials[i]);
    fprintf(log_performance,"\n");
  }  
  perf_mean_value = mean(perf_of_trials,max_trials);
  perf_stdev_value = stdev(perf_of_trials,max_trials,perf_mean_value);
  fprintf(log_performance,"Mean %f\t ",perf_mean_value);
  fprintf(log_performance,"\tStd Dev %f\t ",perf_stdev_value);

  cout << "Mean " << perf_mean_value << " Std " << perf_stdev_value << endl; 
  
  fclose(log_performance);
  if(change_mode==4) fclose(log_varying_values);

  //free memory
  delete[] perf_of_trials;
  delete[] perf_of_iterations; 
}

void statistics_and_output(int r, int t){
  //*NOTE*: change the best_so_far_ant and 
  //n_ants values according to your algorithm 
  //implementation
  performance[r-1][t-1] = best_so_far_ant->tour_length;
  //output varying values when (change_mode=4)
  //the values for all RUNS are identical
  if(change_mode==4 && r==1){
    fprintf(log_varying_values, "%.2f", change_degree);
    fprintf(log_varying_values,"\n");
  }
  cout<<"iteration: " << current_iteration <<  " best_so_far " << best_so_far_ant->tour_length << endl;
}

void apply_to_algorithm(){
  //*NOTE*: These actions stand only for ACO 
  //algorithms. In case an EA is used, simply
  //evaluate the individuals in the population 
  //using fitness_evaluation()
  compute_nn_lists();
  init_heuristic_info();
  compute_total_info();  
  //Reset best solution after a change to
  //calculate the modified offline performance
  best_so_far_ant->tour_length = INFTY;
}

void init_try(int t){
  //initialize algorithmic structures 
  //since the initial objects are added
  //at the beggining of each Run and the 
  //dynamic environments are re-initialized
  //using initialize_environment()
  compute_nn_lists();             //reset nearest neighbour objects
  init_heuristic_info();          //initialize heuristic info
  if(alg_mode == 1){
    trail0 = 1.0 / (rho * nn_tour()); 
  }      
  init_pheromone_trails(trail0);  //initialize pheromone trails
  compute_total_info();           //combine heuristic+pheromone
  best_so_far_ant->tour_length = INFTY; //reset best solution found 
}

void exit_try(){
  //free memory
  if(change_mode ==2 || change_mode==3) {
    for(int i =0; i < cyclic_states; i++){
      delete[] cyclic_random_vector[i];
      delete[] cyclic_re_random_vector[i];
    }
    delete[] cyclic_random_vector;
    delete[] cyclic_re_random_vector;
  }
  if(noise_flag == true){
    delete[] noise;
    delete[] re_noise;
  }
}

void free_memory(){
  //free memory from the DBGP impemenation
  delete[] masked_objects;
  delete[] init_objects;
  delete[] masked_demand; 
  delete[] init_demand;
  for(int i =0; i < problem_size; i++) {
    delete[] distances[i];
  }
  delete[] distances;
  //free memory from the ACO implementation
  delete[] best_so_far_ant->tour;
  delete[] best_so_far_ant->routes;
  for(int i =0; i < n_ants; i++){
    delete[] ant_population[i].tour;
    delete[] ant_population[i].visited;
    delete[] ant_population[i].routes;
  }
  delete[] ant_population;
  for(int i =0; i < problem_size; i++){
    delete[] pheromone[i];
    delete[] heuristic[i];
    delete[] total[i];
    delete[] nn_list[i];
  }
  delete[] pheromone;
  delete[] heuristic; 
  delete[] total;
  delete[] nn_list;
  delete[] prob_of_selection;   
  delete[] perf_filename;
  delete[] var_filename;
  for(int i = 0; i < max_trials; i++) {
    delete[] performance[i];
  }
  delete[] performance;
}
//end ACO

/****************************************************************/
/*                Main Function (DBGP+ACO)                      */
/****************************************************************/
int main(int argc, char *argv[]) {
  // read in parameters
  problem_instance = argv[1];
  change_mode = atoi(argv[2]);
  change_degree = atof(argv[3]);
  change_speed = atoi(argv[4]); 
  
  max_iterations = 1000;
  max_trials = 3;
  
  read_problem(problem_instance);   //Read TSP from file
  check_input_parameters();         //validate the parameters  
  
  open_stats();
  
  //ACO algorithms initialization
  set_algorithm_parameters();
  allocate_ants();
  allocate_structures();
  
  for(int run = 1; run <= max_trials; run++){
    cout<< "-------------Run: " << run << "------------------" << endl; 
    srand(1);                       //static seed reset for the changes
    seed = run;                     //changable seed for the algorithm
    initialize_environment();       //set DBGP for a new Run  
    current_iteration=1;            //set iteration for a new Run
    init_try(run);  

    //re-initialize algorithmic structures
    while(current_iteration <= max_iterations){//termination not met
      //ACO algorithm methods
      construct_solutions();  
      update_best();
      pheromone_update();
      //output results to files
      statistics_and_output(run,current_iteration);     
      // environmental changes (integration of ACO with DBGP)  
      if(current_iteration%change_speed == 0){  
	 change_environment(current_iteration);	
	 apply_to_algorithm();
      }
      current_iteration++;
    }  
    exit_try(); 
  }
  close_stats();
  
  free_memory();
  return 0;
}
