/**************************************************************************************/
/* GA4DOP.c                                                                           */
/* This is an implementation of different genetic algorithms for dynamic              */
/* optimisation problems with the XOR generator for the following paper:              */
/*                                                                                    */ 
/*   S. Yang. Genetic algorithms with memory- and elitism-based immigrants in         */
/*   dynamic environments. Evolutionary Computation, 16(3): 385-416, Fall 2008.       */
/*   The MIT Press (DOI: 10.1162/evco.2008.16.3.385).                                 */
/*                                                                                    */
/* Compile:                                                                           */
/*   g++ -o GA4DOP GA4DOP.c                                                           */
/* Run:                                                                               */
/*   ./GA4DOP gaMode functNo chgMode chgDeg chgSpeed immRatio pImmMut pBaseMut rSeed  */
/*                                                                                    */
/* Written by:                                                                        */
/*   Shengxiang Yang, University of Leicester, UK, May 2005; refined November 2009    */
/*                                                                                    */
/* If any query, please email Shengxiang Yang at s.yang@mcs.le.ac.uk                  */ 
/*                                                                                    */
/**************************************************************************************/

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string>
#include <fstream>

using namespace std;

/* Change any of these parameters to match your needs */
#define NGENES 100       // no. of genes in a binary chromosome
#define MAXPOPSIZE 120   // max population size
#define MEMSIZE 10       // memory population size
#define MAXBASESTATES 20 // total base states for cyclical changes

/* Global variables */
FILE *logPopBest;     // output file for best-of-pop fitness over generations
FILE *logPopDiv;      // output file for pop diversity
int currentGenNo=0;   // current generation no

double pMutation = 0.01;    // prob. of mutation
double pCrossover = 0.6;    // prob. of crossover
double pParamUX = 0.5;      // swapping probability for parameterized UX
double immigrateRatio =0.2; // ratio of random immigrants to total population size
int nImmigrants;            // number of random immigrants
double pImmMut = 0.01;      // prob. of mutating best memory point for memory based 
                            // immigration or mutating the elite for elitism based 
                            // immigration

int randSeed = 0;
int gaMode = 1; // GA mode (with 1 elite):  1-SGA, 2-RIGA, 3-MEGA, 4-MRIGA, 5-MIGA
                //  6-EIGA, 7-MSGA, 8-ISGA with best memory point and dynamic gene 
                //  pool (ECJ paper)

int functNo = 1;  // Function: 1-DUF(25,4,1), 2-DUF(25,4,3), 3-DUF(25,4,4), 
                  //  4-DUF(25,4,deceptive), 25 copies of 4-bit deceptive problems
                  //  5-Knapsack Problem

int changeMode = 1; // how environment changes, cyclical or not
                    //   1: cyclic, 2: cyclic with noise, 3: random
double changeDegree = 0.5; // the ratio of loci changed in dynamic environment
int changeSpeed = 10;      // the period of environment changing in generations
int totalChanges = 200;    // the total number of environment changes in a run
bool changeDetected;       // whether the enviroment change has been detected

int xorMask[NGENES];    // the mask for XORing environment
bool bRandRatio =false; // whether the changeDegree is random (default to false)
int baseXORMasks[MAXBASESTATES][NGENES]; // base xor masks for cyclical environment
int totalBaseStates;
double pBaseMut = 0.05; // probability of mutating each element of a base XOR mask 

int popSize;      // actual pop size (including memory if used)
int popSize1;     // size of main or 1st population
int popSize2;     // size of 2nd population
int newPopSize1;  // size of 1st population in next generation
int newPopSize2;  // size of 2nd population in next generation
int minPopSize1;  // min size of 1st population
int minPopSize2;  // min size of 2nd population
int maxPopSize1;  // max size of 1st population
int maxPopSize2;  // max size of 2nd population
int deltaPopSize; // amount of pop size to be adjusted for two populations

int memSize;           // memory size
int memLevel;          // number of points in the memory (full if memLevel == memSize) 
int nextMemUpdateTime; // next memory update generation for memory enhanced GAs, memory 
                       //  updated after next random number of generations in [5, 10]

struct genotype {      // genotype (GT), a member of the population
  int gene[NGENES];       
  int phenoGene[NGENES];  // phenotype allele for dynamic environment

  double fitness;         // phenotype fitness
  double rfitness;        // relative fitness
  double sfitness;        // scaled fitness
  double cfitness;        // cumulative fitness
  double oldfitness;      // fitness of previous generation for memory points
                          // used for detecting environmental changes 
};

struct genotype pop[MAXPOPSIZE+1];    // (max) population
struct genotype newpop[MAXPOPSIZE+1]; // (max) temporary population
struct genotype tmpMember;            // temporary individual for sortPopulation()

#define numSegs 200
#define maxSegLength 50
#define minSegLength 5
#define fixSegLength 5
int geneSegPool[numSegs][fixSegLength]; // the gene segment pool
int geneSegPosition[numSegs];           // the starting locus for each gene segment
int geneSegLength[numSegs];             // the length for each gene segment
bool bRandomSeg[numSegs];               // whether a segment is random or not
double pTransform = 0.9;                // transformation rate for ISGA
double replaceRate = 0.2;               // percentage of gene segments to update   
double randomSegRatio = 0.3;            // percentage of random gene segments
int randomSegStartIdx;                  
int totalReplaces; 

/***************************************************/
/* Utility Functions                               */
/***************************************************/
int randomBit() { return (rand()%2); }

double randNumber(double low, double high) {
  return ((double)(rand()%10000)/10000.0)*(high - low) + low;  
}

int sign(double value) {
  if (value == 0.0) return 0; 
  else return (value > 0.0? 1 : -1);
}

template <typename T>
void swap(T *x, T *y) {
  T temp;
  temp = *x; *x = *y; *y = temp;
}

/***********************************************************/
/* Random sequence generator:                              */
/*   Generates a random sequence of seqSize numbers from   */
/*   the natural number array [0, 1, ..., numSize-1]       */
/***********************************************************/
void randomSequence(int* seqArr, int seqSize, int numSize) {
  int i, j, idx, count, numleft;

  // constructs a natural number array
  int number[numSize];
  for (i=0; i<numSize; ++i) number[i] = i; 

  // select seqSize numbers from number[] without repeat
  numleft = numSize;
  for (i=0; i<seqSize; ++i) { 
    idx = rand()%numleft;
    count = 0;
    for (j=0; j<numSize; ++j) { 
      if (number[j] != -1) // found one not selected number
        ++count;

      if (count > idx) break; // found the idx-th not selected number
    }

    seqArr[i] = number[j];
    number[j] = -1; // marked as selected
    --numleft;
  }
}

/******************************************************/
/* Decomposable Unitation-based Functions (DUFs)      */
/******************************************************/
#define NBLOCKS 25    // the number of schemas
#define BLOCKSIZE 4   // block size - length of target schema

void DUF(int mem) {
  int i, j, countOFOnes;

  for (i = 0; i < NGENES; ++i) // perform XORing operation
    pop[mem].phenoGene[i] = pop[mem].gene[i] ^ xorMask[i];

  pop[mem].fitness = 0.0;
  for (i = 0; i < NBLOCKS; ++i) { 
    countOFOnes=0;
    for (j = 0; j < BLOCKSIZE; j++)
      if (pop[mem].phenoGene[i*BLOCKSIZE+j] == 1)
        countOFOnes++;  // count the 1 bits in the block

    switch(functNo) {
      case 1: // NK(25,4,1)
        pop[mem].fitness += countOFOnes; 
        break;
      case 2: // NK(25,4,2)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else if(countOFOnes == 3) pop[mem].fitness += 2;
        break;
      case 3: // NK(25,4,4)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        break;
      case 4: // NK(25,4,deceptive)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else pop[mem].fitness += (3 - countOFOnes);
        break;
    }
  }
}

/*************************************************************/
/* Knapsack Problem: select items to fill a knapsack to      */
/*    maximize profits, defined as follows:                  */
/*                                                           */
/*    f(X)= sum_i (p_i*x_i)                                  */
/*    subject to:                                            */
/*        sum_i (w_i*x_i) <= CAPACITY                        */
/*                                                           */
/*************************************************************/
int weight[NGENES];
int profit[NGENES];
int CAPACITY, TOTALWEIGHT;
double capFactor = 0.5;

void Knapsack(int mem) {
  int i, j;
  double sumW, sumP;
  double pFactor = 0.00001;   // punish factor

  for (i = 0; i < NGENES; ++i) // perform XORing operation
    pop[mem].phenoGene[i] = pop[mem].gene[i] ^ xorMask[i];

  pop[mem].fitness = 0.0;
  sumW = sumP = 0.0;
  for (i = 0; i < NGENES; i++) {
    if (pop[mem].phenoGene[i] == 1) {
      sumW += (double)weight[i];
      sumP += (double)profit[i];
    }
  }

  if (sumW > CAPACITY)
    pop[mem].fitness = pFactor *(TOTALWEIGHT - sumW);
  else pop[mem].fitness = sumP;
}

/***************************************************************/
/* Initialization function: Initializes the values of genes    */
/* in binary. It also initializes (to zero) all fitness values */
/* for each member of the main and memory populations. It      */
/* initializes the percentage of 1's to be 50% for each gene   */
/* locus.                                                      */
/***************************************************************/
void initialize() {
  int i, j;

  // intitialize the mask to static problem
  for (i=0; i<NGENES; ++i) xorMask[i] = 0;

  switch(gaMode) {
    case 1:  // SGA
      popSize = MAXPOPSIZE;
      popSize1 = popSize;
      newPopSize1 = popSize1;
      memSize = 0;
      break;
    case 2:  // RIGA
    case 6:  // EIGA
      popSize = (int)ceil((double)MAXPOPSIZE/(1.0 + immigrateRatio));
      popSize1 = popSize;
      newPopSize1 = popSize1;
      memSize = 0;
      break;
    case 3:  // MEGA
      popSize = MAXPOPSIZE;
      popSize1 = (int)(0.9*popSize);
      newPopSize1 = popSize1;
      memSize = (int)(0.1*popSize);
      memLevel = 0;
      break;
    case 4:  // MRIGA
    case 5:  // MIGA
      popSize = (int)ceil((double)MAXPOPSIZE/(1.0 + immigrateRatio));
      memSize = (int)(0.1*popSize);
      popSize1 = popSize - memSize;
      newPopSize1 = popSize1;
      memLevel = 0;
      break;
    case 7:  // MSGA
      popSize = MAXPOPSIZE;
      popSize1 = (int)(0.45*popSize);
      popSize2 = (int)(0.45*popSize);
      newPopSize1 = popSize1;
      newPopSize2 = popSize2;
      memSize = (int)(0.1*popSize);
      minPopSize1 = (int)(0.3*popSize);
      minPopSize2 = (int)(0.3*popSize);
      maxPopSize1 = (int)(0.6*popSize);
      maxPopSize2 = (int)(0.6*popSize);
      deltaPopSize = (int)(0.05*popSize); 
      memLevel = 0;
      break;
    case 8: // ISGA
      popSize = MAXPOPSIZE;
      memSize = (int)(0.1*popSize);
      popSize1 = popSize - memSize;
      newPopSize1 = popSize1;
      memLevel = 0;
      break;
  }

  // initialize the whole population including memory if used
  for (i=0; i<=MAXPOPSIZE; ++i) 
    for (j=0; j<NGENES; ++j) {
        pop[i].gene[j] = randomBit();
        pop[i].phenoGene[j] = pop[i].gene[j];
    }

  for (i=0; i<=MAXPOPSIZE; ++i) {
    pop[i].fitness = 0.0;
    pop[i].rfitness = 0.0;
    pop[i].sfitness = 0.0;
    pop[i].cfitness = 0.0;
    pop[i].oldfitness = 0.0;
  }

  if (gaMode == 8) { // ISGA
    // initialize gene segments randomly
    for (i=0; i<numSegs; ++i) {
      // int length = rand()%(maxSegLength-minSegLength+1) + minSegLength;
      // geneSegLength[i] = length;
      // geneSegPosition[i] = rand()%(NGENES-length);
      geneSegLength[i] = fixSegLength;
      geneSegPosition[i] = rand()%(NGENES-fixSegLength);

      for (j=0; j<fixSegLength; ++j) 
        geneSegPool[i][j] = randomBit();
    }

    randomSegStartIdx = (int)((1.0 - randomSegRatio)*numSegs);

    totalReplaces = 140;
    for (i=0; i<numSegs; ++i) bRandomSeg[i] = true;
  }

  if (functNo == 5) {
    int totalW = 0;
    ifstream ifs("Knapsack100.txt",ios::in);
    if (!ifs) {
      cerr << "Error: unable to open the data file." << endl;
      exit(1);
    }

    char temp[20];
    for (i = 0; i < NGENES; i++) {
      ifs >> temp >> weight[i] >> temp >> profit[i];
      totalW += weight[i];
    }

    TOTALWEIGHT = totalW;
    CAPACITY = (int)(capFactor * TOTALWEIGHT);
  }

  nImmigrants = (int)(immigrateRatio * popSize);
  changeDetected = false;
  
cout << "popSize: " << popSize << " popSize1: " << popSize1 
     << " popSize2: " << popSize2
     << " memSize: " << memSize << " nImmigrants: " << nImmigrants << endl;
}

/***********************************************************/
/* Sort sub-population using bubble-sorting algorithm in   */ 
/* the order of decreasing fitness                         */
/***********************************************************/
void sortPopulation(int start, int end) {
  int i, j;
  for (i = start; i < end; ++i) 
    for (j = start + 1; j < (start + end - i); ++j)
      if (pop[j-1].fitness < pop[j].fitness) {
        tmpMember = pop[j-1];
        pop[j-1] = pop[j];
        pop[j] = tmpMember;
      }
}

// evaluat the main population, excluding memory if used
void evaluatePop() {
  if (functNo == 5) {
    for (int mem=0; mem < popSize - memSize; ++mem) 
      Knapsack(mem);
  }
  else {
    for (int mem=0; mem < popSize - memSize; ++mem) 
      DUF(mem);
  }
}

// evaluate memory and detect environmental changes
void evaluateMemory() {
  int i, memStartID = popSize - memSize;
  for (i = memStartID; i < popSize; ++i) // store old fitness
    pop[i].oldfitness = pop[i].fitness;

  changeDetected = false; 
  if (functNo == 5) {
    for (i=memStartID; i<popSize; ++i) {
      Knapsack(i);

      // if not the initial generation, check whether change occurs
      if (currentGenNo != 0) {  
        if (pop[i].oldfitness != pop[i].fitness)
          changeDetected = true;
      }
    }
  }
  else {
    for (i=memStartID; i<popSize; ++i) {
      DUF(i);

      // if not the initial generation, check whether change occurs
      if (currentGenNo != 0) {  
        if (pop[i].oldfitness != pop[i].fitness)
          changeDetected = true;
      }
    }
  }
}

void storeElite() {
  int idx;
  switch(gaMode) {
    case 1: case 2: case 3: case 4: case 5:
    case 6: case 8: 
      sortPopulation(0, popSize1);  // sort population
      idx = 0;
      break;
    case 7: 
      sortPopulation(0, popSize1);  // sort population 1 
      sortPopulation(popSize1, popSize1+popSize2);  // sort population 2
      idx = (pop[0].fitness > pop[popSize1].fitness) ? 0 : popSize1;
      break;
    default: break;
  }

  pop[popSize] = pop[idx]; // store elite
}

// replace the worst individual in pop 1 with elite from last population 
// that is re-evaluated
void retrieveElite() {

  // re-evaluate the elite
  if (functNo==5) Knapsack(popSize);
  else DUF(popSize);

  sortPopulation(0, popSize1);  // sort pop 1
  pop[popSize1-1] = pop[popSize];
}

/**************************************************************/
/* Adjust pop sizes: Adjust sizes for pops according to their */
/*     performance. Winner wins deltaPopSize from loser.      */
/**************************************************************/
void adjustPopSize() {
  double bestFitness1 = pop[0].fitness;
  double bestFitness2 = pop[popSize1].fitness;

  for (int i = 1; i < popSize1; ++i)
    if (bestFitness1 < pop[i].fitness)
      bestFitness1 = pop[i].fitness;

  for (int i = popSize1+1; i < popSize1+popSize2; ++i) 
    if (bestFitness2 < pop[i].fitness)
      bestFitness2 = pop[i].fitness;

  if (bestFitness1 > bestFitness2)
    newPopSize1 = min(popSize1 + deltaPopSize, maxPopSize1);
  else if (bestFitness1 < bestFitness2)
    newPopSize1 = max(popSize1 - deltaPopSize, minPopSize1);
  newPopSize2 = popSize - newPopSize1 - memSize;
}

/***************************************************************/
/* calcCumulateFitness: calculate the cumulative fitness of a  */
/*   pop using the sigma truncation scaling method descibed in */
/*   goldberg p 124. If the scaled fitness is less than zero,  */
/*   it is arbitrarily set to zero (thus the truncation part   */
/*   of 'sigma truncation').                                   */
/***************************************************************/
void calcCumulateFitness(int start, int end) {
  // find total fitness of the population      
  double sum = 0.0;
  for (int i = start; i < end; ++i)
    sum += pop[i].fitness;

  // calculate relative fitness
  if (sum == 0.0) { // all members have 0 fitness
    for (int i = start; i < end; ++i)
      pop[i].rfitness = 1.0/(end - start);  // set relative fitness
  }
  else {
    for (int i = start; i < end; ++i) // calculate relative fitness
      // pop[i].rfitness = pop[i].sfitness/sum;
      pop[i].rfitness = pop[i].fitness/sum;
  }

  // calculate cumulative fitness
  pop[start].cfitness = pop[start].rfitness;
  for (int i = start + 1; i < end; ++i)
    pop[i].cfitness =  pop[i-1].cfitness + pop[i].rfitness;
}

/**************************************************************/
/* Selection function: Standard proportional selection for    */
/* maximization problems                                      */
/**************************************************************/
void select() {
  int i, j, mem, index;
  double ptr, p, sum = 0.0;

  // adjusts next pop sizes for two pops (MSGA)
  if (gaMode == 7) adjustPopSize(); 

  /**********************************************************  
   * Using standard fitness proportional selection without  *
   *    Sigma Trunction Scaling                             *
   **********************************************************/
  // calculate cumulative fitness for pop 1
  calcCumulateFitness(0, popSize1);

  // select newPopSize1 survivors using cumulative fitness.
  for (i=0; i<newPopSize1; ++i) { 
    p = randNumber(0.0, 1.0);
    if (p < pop[0].cfitness) newpop[i] = pop[0];      
    else {
      for (j=0; j<popSize1; ++j)
        if (p >= pop[j].cfitness && p<pop[j+1].cfitness)
          { newpop[i] = pop[j+1]; break; }
    }
  }

  if (gaMode == 7) {
    // calculate cumulative fitness for pop 2
    calcCumulateFitness(popSize1, popSize1+popSize2);

    // select newPopSize2 survivors using cumulative fitness.
    for (i=newPopSize1; i<(popSize - memSize); ++i) { 
      p = randNumber(0.0, 1.0);
      if (p < pop[popSize1].cfitness) newpop[i] = pop[popSize1];      
      else {
        for (j=popSize1; j<(popSize - memSize); ++j)
          if (p >= pop[j].cfitness && p<pop[j+1].cfitness)
            { newpop[i] = pop[j+1]; break; }
      }
    }

    // update pop size variables
    popSize1 = newPopSize1;
    popSize2 = newPopSize2;
  }

  /* once a new population is created, copy it back */
  for (i=0; i<(popSize - memSize); ++i)
    pop[i] = newpop[i];
}

/**************************************************************/
/* Uniform Crossover: Parameterized                           */
/**************************************************************/
void UniformXover(int one, int two) {
  if (NGENES > 2) {
    for (int i=0; i<NGENES; ++i)
      if (randNumber(0.0, 1.0) < pParamUX) { // pParamUX = 0.5
        if (pop[one].gene[i] != pop[two].gene[i])
          swap(pop[one].gene[i], pop[two].gene[i]);
      }
  }
}

/***************************************************************/
/* Crossover selection: selects two parents that take part in  */
/* the crossover. Implements uniform crossover                */
/***************************************************************/
void crossover(int start, int end) {
  int i, j, k;
  int psize = end - start;

  /* shuffle the population */
  int index[psize];  // random index array
  randomSequence(index, psize, psize);

  for (i=0; i<psize; ++i) newpop[start+i] = pop[start+i];

  for (i=0; i<psize; ++i) pop[start+i] = newpop[start+index[i]];

  int mem, one;
  int first = 0;   // count of the number of members chosen
  double p;
  for (mem = start; mem<end; ++mem) {
    p = randNumber(0.0, 1.0);
    if (p < pCrossover) {
      ++first;
      if (first%2 == 0) UniformXover(one, mem); 
      else one = mem;
    }
  }
}

/**************************************************************/
/* Update Gene Pool: update gene segments from the pop member */
/*   randomly                                                 */
/**************************************************************/
void updateGenePool(void) {
  int i, j, one, two, winner, length, pos;

  // set the random gene segments
  for (i=randomSegStartIdx; i<numSegs; ++i) {
    // length = rand()%(maxSegLength-minSegLength+1) + minSegLength;
    // geneSegLength[i] = length;
    // geneSegPosition[i] = rand()%(NGENES-length);
    // for (j=0; j<length; ++j) 
    //   geneSegPool[i][j] = randomBit();

    geneSegPosition[i] = rand()%(NGENES-fixSegLength);
    for (j=0; j<fixSegLength; ++j) 
      geneSegPool[i][j] = randomBit();
  }
  
  // update totalReplaces non-random gene segments from the current population
  int index[totalReplaces];
  randomSequence(index,totalReplaces,randomSegStartIdx);

  for (i=0; i<totalReplaces; ++i) {
    // tournament select one member from which to update a gene segment    
    one = rand()%popSize1;       // one in [0, popSize1-1]
    two = rand()%(popSize1 - 1); // two in [0, popSize1-2]
    if (two >= one) two++;
    winner = (pop[one].fitness > pop[two].fitness ? one : two);

    pos = rand()%(NGENES-fixSegLength);
    geneSegPosition[index[i]] = pos; 

    for (j=0; j<fixSegLength; ++j) 
      geneSegPool[index[i]][j] = pop[winner].gene[pos+j];
  }
}

/*************************************************************/
/* Transformation: select one gene segment from the library  */
/*   randomly and transform it into a member from a random   */  
/*   position                                                */
/*************************************************************/
void transform(int mem) {
  int i, j, stop, segIdx, pos;

  if (randNumber(0.0, 1.0) < pTransform) {
    segIdx = rand()%numSegs;

    if (bRandomSeg[segIdx] == true) 
      pos = rand()%NGENES;
    else // aligned transformation
      pos = geneSegPosition[segIdx];

    stop = min(NGENES, pos+geneSegLength[segIdx]);
    for (i=pos,j=0; i<stop; ++i,++j) 
      pop[mem].gene[i] = geneSegPool[segIdx][j];
  }
}

/*****************************************************/
/* Clone: clone the population                       */
/*****************************************************/
void clone(void) {
  for (int i=0; i<(popSize - memSize); ++i)
    transform(i);
}

/**************************************************************/
/* Mutation: Random uniform mutation. A bit selected for      */
/*           mutation is flipped                              */
/**************************************************************/
void mutate(double pmut) {
  double p;
  for (int i=0; i<(popSize - memSize); ++i) {
    for (int j=0; j<NGENES; ++j) {
      p = randNumber(0.0, 1.0);
      if (p < pmut)
        pop[i].gene[j] = ((pop[i].gene[j]==0)? 1 : 0);
    }
  }
}

/*********************************************************/
/* Restart pop 2 when an environmental change is deteced */
/*********************************************************/
void restart() {
  if (gaMode ==7) { // initialize the 2nd population
    for (int i=popSize1; i<(popSize1+popSize2); ++i) {
      for (int j=0; j<NGENES; ++j) {
        pop[i].gene[j] = randomBit();
        pop[i].phenoGene[j] = pop[i].gene[j];
      }

      // evaluate the fitness
      if (functNo == 5) Knapsack(i);
      else DUF(i);

      pop[i].oldfitness = 0.0;
    }
  }
}

/*******************************************************/
/* Immigration: immigrates replace random individuals  */
/*              in the population                      */
/*******************************************************/
void randomImmigrate(void) {
  int i, j, mem;
  int memStartIdx = popSize - memSize;

  // create a random sequence index array
  int psize = popSize - memSize;
  int index[nImmigrants];  // random index array
  randomSequence(index, nImmigrants, psize);

  // now do immigration
  switch(gaMode) {
    case 2:  // RIGA
    case 4:  // MRIGA
      for (mem=0; mem<nImmigrants; ++mem) { // random immigrate
        for (i=0; i<NGENES; ++i)
          pop[index[mem]].gene[i] = randomBit();
      }
      break;
    case 5:  // MIGA
      if (memLevel != 0) {
        sortPopulation(memStartIdx, memStartIdx+memLevel);  // sort memory
        pop[index[0]] = pop[memStartIdx]; // keep the best of memory

        // mutate from the best of memory into population
        for (mem=1; mem<nImmigrants; ++mem) {
          for (i=0; i<NGENES; ++i) {
            if (randNumber(0.0, 1.0) < pImmMut) 
              pop[index[mem]].gene[i] = 1 - pop[memStartIdx].gene[i]; 
            else  
              pop[index[mem]].gene[i] = pop[memStartIdx].gene[i]; 
          }
        }
      }
      else {  // random immigrate
        for (mem=0; mem<nImmigrants; ++mem) {
          for (i=0; i<NGENES; ++i)
            pop[index[mem]].gene[i] = randomBit();
        }
      }
      break; 
    case 6:  // EIGA
      for (mem=0; mem < nImmigrants; ++mem) { // elitism based immigrates
        for (i=0; i<NGENES; ++i) {
          if (randNumber(0.0, 1.0) < pImmMut) // mutate from the elite
            pop[index[mem]].gene[i] = 1 - pop[popSize].gene[i]; 
          else  
            pop[index[mem]].gene[i] = pop[popSize].gene[i];
	}
      }
      break;
    default: break;
  }
}

/*********************************************************/
/* Immigration: immigrates replace the worst individuals */
/*              in the population                        */
/*********************************************************/
void immigrate(void) {
  int i, j, mem, index;
  int memStartIdx = popSize - memSize;

  sortPopulation(0, popSize1);  // sort pop 1 

  // now do immigration
  switch(gaMode) {
    case 2:  // RIGA
    case 4:  // MRIGA
      for (mem=0; mem < nImmigrants; ++mem) { // random immigrates
        index = popSize1 - 1 - mem;
        for (i=0; i<NGENES; ++i)
          pop[index].gene[i] = randomBit();
      }
      break;
    case 5:  // MIGA
      if (memLevel != 0) {
        sortPopulation(memStartIdx, memStartIdx+memLevel);  // sort memory
        pop[popSize1-1] = pop[memStartIdx]; // keep the best of memory

        // mutate from the best of memory into population
        for (mem=1; mem < nImmigrants; ++mem) {
          index = popSize1 - 1 - mem;
          for (i=0; i<NGENES; ++i) {
            if (randNumber(0.0, 1.0) < pImmMut)
              pop[index].gene[i]=1 - pop[memStartIdx].gene[i]; 
            else
              pop[index].gene[i] = pop[memStartIdx].gene[i]; 
          }
        }
      }
      else {  // random immigrate
        for (mem=0; mem < nImmigrants; ++mem) {
          index = popSize1 - 1 - mem;
          for (i=0; i<NGENES; ++i)
            pop[index].gene[i] = randomBit();
        }
      }
      break;
    case 6:  // EIGA
      for (mem=0; mem < nImmigrants; ++mem) { // elitism based immigrates
        index = popSize1 - 1 - mem;
        for (i=0; i<NGENES; ++i) {
          if (randNumber(0.0, 1.0) < pImmMut) // mutate from the elite
            pop[index].gene[i] = 1 - pop[popSize].gene[i]; 
          else  
            pop[index].gene[i] = pop[popSize].gene[i];
	}
      }
      break;
    default: break;
  }
}

void retrieveMemory() {
  int i, j, index;
  if (gaMode < 8) {
    for (i=0; i<popSize; ++i)   // copy pop + memory
      newpop[i] = pop[i];

    if (popSize > popSize1 + memSize) {  // 2 pops
      j = popSize1;
      for (i=popSize - memSize; i<popSize; ++i, ++j)  // shift memory upward
        pop[j] = pop[i];
    }

    sortPopulation(0, popSize1 + memSize);  // sort pop + memory points 

    // select popSize1 best individuals as the new pop 1
    for (i=0; i<popSize1; ++i) 
      newpop[i] = pop[i];

    // copy the whole pop back without changing the memory
    for (i=0; i<popSize; ++i)
      pop[i] = newpop[i];
  }
  else { // ISGA
    if (memLevel != 0) {
      int memIdx = popSize - memSize; // index of memory point to retrieve
      sortPopulation(memIdx, memIdx+memLevel);  // sort memory
      
      // clone from the best memory point into population
      sortPopulation(0, popSize1);   // sort pop 1 
      pop[popSize1-1] = pop[memIdx]; // keep the best of memory
      for (i=1; i<nImmigrants; ++i) {
        index = popSize1-1-i;
        for (j=0; j<NGENES; ++j) 
          pop[index].gene[j] = pop[memIdx].gene[j];
        transform(index);
      }
    }
    else {  // random immigrate
      for (i=0; i<nImmigrants; ++i) {
        index = popSize1-1-i;
        for (j=0; j<NGENES; ++j)
          pop[index].gene[j] = randomBit();
        transform(index);
      }
    }
  }
}

/******************************************************/
/* Retrieve best individual from main population to   */
/* replace closest memory point                       */
/******************************************************/
void updateMemory(void) {
  int i, j;
  int bestindex = popSize1; // index of the best individual of population
  int memindex;             // index of the memory point to be changed
  int tmpDist, minHammingDist = NGENES;
  int memStartIdx = popSize - memSize;

  switch(gaMode) {
    case 3: case 4: case 5:
      sortPopulation(0, popSize1);
      bestindex = 0;
      break;
    case 7:
      sortPopulation(0, popSize1);
      sortPopulation(popSize1, popSize1+popSize2);
      bestindex = (pop[0].fitness >= pop[popSize1].fitness)? 0 : popSize1;
      break; 
    default: break;
  }

  // if memory not full, store as new point; otherwise, replace the closest
  if (memLevel < memSize) {
    pop[memStartIdx + memLevel] = pop[bestindex];
    memLevel++;
  }
  else { // memory is full
    // find the memory point closest to the best individual of pop
    memindex = memStartIdx;
    for (i = memStartIdx; i < popSize; ++i) {
      tmpDist = 0;
      for (j=0; j<NGENES; ++j) { // calculate edit distance
        if (pop[i].gene[j] != pop[bestindex].gene[j]) 
          tmpDist++;
      }

      if (tmpDist < minHammingDist) {
        minHammingDist = tmpDist; 
        memindex = i;
      }  
    }

    // update memory point if the best individual of pop is better
    if (pop[memindex].fitness < pop[bestindex].fitness) 
      pop[memindex] = pop[bestindex];
  }
}

/*********************************************************/
/* Environment Dynamics Function: Construct base XORing  */
/* masks for (partially) cyclical changing environments  */
/*********************************************************/
void constructBaseXORMasks(void) {
  int i, j, numOnes;
  int randIndex[NGENES];

  // create a random permutation of 0 to NGENES-1
  randomSequence(randIndex, NGENES, NGENES);

  // initialize the baseXORMasks
  for (i=0; i<totalBaseStates; ++i)
    for (j=0; j<NGENES; ++j) baseXORMasks[i][j] = 0;

  // configure the baseXORMasks
  numOnes = (int)(changeDegree * NGENES);
  for (i=0; i<totalBaseStates; ++i) {
    for (j=0; j<numOnes; ++j)
      baseXORMasks[i][randIndex[numOnes*i+j]] = 1; 
  }
}

/*******************************************************/
/* Environment Dynamics Function: Incremental changing */
/*******************************************************/
void changeEnvironment(void) {
  int i, j, numOnes, baseStateIdx;
  int intermMask[NGENES];  // intermediate mask

  for (i=0; i<NGENES; ++i) intermMask[i] = 0; 

  if ( bRandRatio == true)
    changeDegree = randNumber(0.0, 1.0);
  numOnes = (int)(changeDegree * NGENES);

  switch(changeMode) {
    case 1: // cyclical change
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i)  // copy the relevant base XORing mask 
        intermMask[i] = baseXORMasks[baseStateIdx][i];
      break;
    case 2: // partially cyclical change
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i) // copy the relevant base XORing mask
        intermMask[i] = baseXORMasks[baseStateIdx][i];

      for (i=0; i<NGENES; ++i)
        if (randNumber(0.0, 1.0) < pBaseMut)
          intermMask[i] = 1 - intermMask[i];
      break;
    case 3: // random non-cyclical change
      int index2[numOnes];
      randomSequence(index2, numOnes, NGENES);
      for (i=0; i<numOnes; ++i) // create numOnes 1's in intermMask[]
        intermMask[index2[i]] = 1;
      break;
  }

  for (i=0; i<NGENES; ++i) // integrate intermMask[] into xorMask[]
    xorMask[i] ^= intermMask[i];
}

/***************************************************************/
/* Report function: Reports progress of the simulation. Data   */
/* dumped into the output file are separated by " ".           */
/***************************************************************/
void report(void) {
  int idx = 0;
  double popBestFitness=0.0; // best population fitness
  for (int i=0; i<popSize; ++i) { // including memory
    if(popBestFitness < pop[i].fitness) {
      popBestFitness = pop[i].fitness;
      idx = i;
    }
  }

  fprintf(logPopBest, "%d ", (int)popBestFitness);
}

/***************************************************************/
/* Report function: Reports diversity of the population. Data  */
/* dumped into the output file are separated by " ".           */
/***************************************************************/
void reportDiversity(void) {
  int i, j, k, hammingDist;
  double popDiversity = 0.0; 
  for (i = 0; i<popSize; ++i) 
    for (j = 0; j<popSize; ++j) {
      hammingDist = 0;
      for (k = 0; k<NGENES; ++k) {
        if (pop[i].gene[k] != pop[j].gene[k]) hammingDist += 1;
      }
      popDiversity += (double)hammingDist;
    }

  popDiversity /= (double)(NGENES*popSize*(popSize-1));

  fprintf(logPopDiv, "%4.3f ", popDiversity);
}

/**************************************************************/
/* Main function: Each generation involves selecting the best */
/* members, performing crossover & mutation and then          */
/* evaluating the resulting population, until the terminating */
/* condition is satisfied                                     */
/**************************************************************/
int main(int argc, char *argv[]) {
  // read in parameters
  gaMode = atoi(argv[1]);
  functNo = atoi(argv[2]);
  changeMode = atoi(argv[3]);
  changeDegree = atof(argv[4]);
  changeSpeed = atoi(argv[5]);
  immigrateRatio = atof(argv[6]);
  pImmMut = atof(argv[7]);
  pBaseMut = atof(argv[8]);
  randSeed = atoi(argv[9]);
  srand(randSeed);

  if (changeDegree > 1.0) bRandRatio = true;
  else bRandRatio = false;

  if (changeMode == 1 || changeMode == 2) {
    if(changeDegree == 0.1)  totalBaseStates = 10;
    if(changeDegree == 0.2)  totalBaseStates = 5;
    if(changeDegree == 0.5)  totalBaseStates = 2;
    if(changeDegree == 1.0)  totalBaseStates = 1;
    constructBaseXORMasks();
  }

  bool recordDiversity = false;
  if (changeSpeed == 50 && immigrateRatio == 0.2 
      && pImmMut == 0.01 && pBaseMut == 0.05) 
    recordDiversity = true;

  char * filename = new char[60];
  if (bRandRatio == false)
    sprintf(filename, "PopBest_GA%d_F%d_%d_%4.2f_%d_%4.2f_%4.3f_%4.2f.txt",
            gaMode, functNo, changeMode, changeDegree, changeSpeed, immigrateRatio, 
            pImmMut, pBaseMut);
  else 
    sprintf(filename, "PopBest_GA%d_F%d_%d_Rand_%d_%4.2f_%4.3f_%4.2f.txt",
            gaMode, functNo, changeMode,changeSpeed, immigrateRatio, 
            pImmMut, pBaseMut);

  if ((logPopBest = fopen(filename,"a")) == NULL) { exit(2); }
  fprintf(logPopBest,"%d ", randSeed);

  if (recordDiversity == true) {
    char * divfilename = new char[60];
    sprintf(divfilename, "PopDiv_GA%d_F%d_%d_%4.2f_%d_%4.2f_%4.3f_%4.2f.txt", 
            gaMode, functNo, changeMode, changeDegree, changeSpeed, immigrateRatio, 
            pImmMut, pBaseMut);

    if ((logPopDiv = fopen(divfilename,"a")) == NULL) { exit(3); }
    fprintf(logPopDiv,"%d ", randSeed); 
  }

  cout << "The program GA" << gaMode <<" for function" << functNo
       << " is running with random seed " << randSeed << endl;
  currentGenNo = 0;
  nextMemUpdateTime = rand()%6 + 5 + currentGenNo;

  initialize();
  currentGenNo = 0;
  while (currentGenNo <= changeSpeed * totalChanges) {
    evaluatePop();

    switch(gaMode) {
      case 3: case 4: case 5: case 7: case 8:
        evaluateMemory(); break;
      default: break;
    }

    if (currentGenNo > 0) retrieveElite();

    if (gaMode == 8) updateGenePool();

    if (changeDetected == true) {
      switch(gaMode) {
        case 3: case 4: case 8:
          retrieveMemory(); break;
        case 7: 
          retrieveMemory(); restart(); break;
        default: break;
      }
    }

    // time to update memory
    if (currentGenNo == nextMemUpdateTime || 
        (currentGenNo%changeSpeed) == changeSpeed-1) {
      if (gaMode > 2 && gaMode != 6) { // GAs with memory
        updateMemory(); 
        nextMemUpdateTime = rand()%6 + 5 + currentGenNo;
      }
    }

    switch(gaMode) { // GAs with immigrants
      case 2: case 4: case 5: case 6:
        immigrate(); break;
      default: break;
    }

    storeElite();
    report();
    if (recordDiversity == true) reportDiversity();

    select();
    switch(gaMode) {
      case 1: case 2: case 3: case 4: 
      case 5: case 6:  
        crossover(0, popSize1);
        mutate(pMutation); break;
      case 7:
        crossover(0, popSize1);
        crossover(popSize1, popSize1+popSize2);
        mutate(pMutation); break;
      case 8:
        clone();
        mutate(pMutation); break;
      default: break;
    }

    if ((currentGenNo%changeSpeed) == 0) // change environment
      changeEnvironment();

    currentGenNo++;
  }

  fprintf(logPopBest, "\n");  // next line for next run
  fclose(logPopBest);

  if (recordDiversity == true) {
    fprintf(logPopDiv, "\n");  // next line for next run
    fclose(logPopDiv);
  }

  cout << "Success!" << endl;
  return 0;
}

/**************************** End *************************/
