/*==========================================================================
//  Implementation of Grid-Based Evolutionary Algorithm (GrEA)
//  Last update 10, Aug, 2013
//
//  Please find details of the method in th following paper
//  S. Yang, M. Li, X. Liu, and J. Zheng. A Grid-Based Evolutionary Algorithm for Many-Objective Optimization. 
//  IEEE Transactions on Evolutionary Computation, vol. 17, no. 5, pp. 721-736, 2013.
//
//  The source code of GrEA was implemented by Miqing Li.  
//
//  The codes are free for reserach work.
//  If you have any problem with the source codes, please contact 
//  Shengxiang Yang at syang@dmu.ac.uk or Miqing Li at miqing.li@brunel.ac.uk
===========================================================================*/

// NOTE: This code is for multiobjective minimization problems; for maximization problems, some procedures need to be modified, 
// such as fitness assignment, Pareto dominance relation, grid dominance relation, and fitness adjustment.

# include <stdio.h>
# include <stdlib.h>
# include <math.h>
# include <time.h>

# include "Global.h"
# include "Random.h"


int main (int argc, char **argv)
{
    int i, j;
	int random;
	double runTime, startTime, endTime;
	
	FILE *fptOutputObj;         // for the output of the final population in the objective space    
    FILE *fptOutputVar;         // for the output of the final population in the variable space
	
    population *parent_pop;
    population *child_pop;
    population *mixed_pop;

	srand((unsigned)time(NULL));
    random = rand()%1000;

    seed=(float) random/1000.0;
    if (seed<=0.0 || seed>=1.0)
    {
        printf("\n Entered seed value is wrong, seed value must be in (0,1) \n");
        exit(1);
    }

    fptOutputObj = fopen("populationObj","w");
    fptOutputVar = fopen("populationVar","w");


/*-------test problem setting------start---------*/
	// now for the DTLZ2 instance; other instances can be chosen when activating them in file "problemdef.cpp"
	nobj=10;				 // number of objectives
	nreal = nobj + 9;		 // number of variable
	ncon=0;                  // number of constraints
	
	// only for the DTLZ5(I,M) problem suite; I_number (nobj >= I_number >= 2) is the actual dimensionality of the Pareto front
	I_number = 5;            
	
	// set the range of the variables
	min_realvar = (double *)malloc(nreal*sizeof(double));
	max_realvar = (double *)malloc(nreal*sizeof(double));
	for (i=0; i<nreal; i++)
	{
		min_realvar[i]=0;
		max_realvar[i]=1;  
	}
/*-------test problem setting------end---------*/


/*-------parameter setting in the algorithm------start---------*/
	popsize=100;             // population size
    neval=30000;             // maximal number of evaluations
	nrun=2;					 // total number of runs
	grid_div=8;              // grid division in GrEA
	
	
	pcross_real=1.0;		 // crossover probability
	pmut_real=1.0/nreal;     // mutation probability
	eta_c=20.0;              // distribution index of simulated binary crossover (SBX)
	eta_m=20.0;			     // distribution index of polynomial mutation
/*-------parameter setting in the algorithm------end---------*/


    parent_pop = (population *)malloc(sizeof(population));
    child_pop = (population *)malloc(sizeof(population));
    mixed_pop = (population *)malloc(sizeof(population));

    allocate_memory_pop (parent_pop, popsize);
    allocate_memory_pop (child_pop, popsize);
    allocate_memory_pop (mixed_pop, 2*popsize);

	grid_min = (double *)malloc(nobj*sizeof(double));             
	grid_max = (double *)malloc(nobj*sizeof(double)); 
	grid_distance = (double *)malloc(nobj*sizeof(double));
	
    randomize();

	printf("Parameter setting done, now start running ");
	
	for (j=0; j<nrun; j++)
	{
		currenteval = 0;
		i=1;
		printf("\n %d run; %d generation...", j+1, i);
		initialize_pop (parent_pop);
		evaluate_pop (parent_pop);

		startTime = clock();

		while (currenteval < neval)
		{
			i++;
			printf("\n %d run; %d generation...", j+1, i);
			construct_grid (parent_pop);				// construct grid environment
			assign_GR_GCPD (parent_pop);				// assign GR and GCPD for individuals in the population
			assign_GCD (parent_pop);					// assign GCD for individuals in the population
			selection (parent_pop, child_pop);			// mating selection and crossover
			mutation_pop (child_pop);					// mutation
			evaluate_pop(child_pop);
			merge (parent_pop, child_pop, mixed_pop);
			environmental_selection (mixed_pop, parent_pop);	// environmental selection		
		}    
		endTime=clock();
		runTime=(endTime-startTime)/CLOCKS_PER_SEC;
		printf("\n the time of the %d run is %f\n", j+1, runTime);
		
		report_variable(parent_pop,fptOutputVar);		// report individuals' value in the decision space
		report_objective(parent_pop,fptOutputObj);		// report individuals' value in the objective space
		
		fflush(stdout);
		fflush(fptOutputVar);
		fflush(fptOutputObj);
	
	}
	fclose(fptOutputVar);
	fclose(fptOutputObj);

    free (min_realvar);
    free (max_realvar);
	free (grid_distance);
	free (grid_max);
	free (grid_min);

    deallocate_memory_pop (parent_pop, popsize);
    deallocate_memory_pop (child_pop, popsize);
    deallocate_memory_pop (mixed_pop, 2*popsize);
    free (parent_pop);
    free (child_pop);
    free (mixed_pop);
    printf("\n Routine successfully exited \a\n");
    return (0);
}

