#ifndef __EVOLUTION_H_
#define __EVOLUTION_H_

#include "..\common\global.h"
#include "..\common\recombination.h"
#include "..\common\common.h"
#include "dmoeaclass.h"


class CMOEAD
{

public:
	CMOEAD();
	virtual ~CMOEAD();

	void init_uniformweight();               // initialize the weights for subproblems
	void init_neighbourhood();               // calculate the neighbourhood of each subproblem
	void init_population();                  // initialize the population
	void reinit_uniformweight();
	void reinit_neighbourhood();
	void TP_response(bool flag);
	void update_reference(CMOEADInd &ind);                 // update ideal point which is used in Tchebycheff or NBI method
	void update_problem(CMOEADInd &ind, int id, int type, int rule); // update current solutions in the neighbourhood
	void update_worstpoint(CMOEADInd &ind);
	void population_match();
	double ASF_function(vector<double> vec1, int n);
	void nondominate_rank_pop();
	void farthest_first_selection();
	void nondominate_truncation();

	void diffevolution();                                      // DE-based recombination
	void matingselection1(vector<int> &list, int cid, int size, int type);  // select mating parents
	
	double nichecount(int cid,int flag);                    // calculate niche count
	void matingselection2(vector<int> &list, int cid, int size, int rule);  // reasonably select mating parents

	// execute MOEAD
	void execute(int run, char *strfunc, char *stralg);

    void read_front(char filename[1024]);
    void calc_distance();

	void save_front(char savefilename[1024]);       // save the pareto front into files
	void save_ps(char savefilename[1024]);

    vector <CSUB>       population;
	vector <CMOEADInd>  ps;
	vector <CSUB>       offspring;
	vector <CMOEADInd>  arch_wp;
	vector <int>        array;
	CMOEADInd           *ind_arr;

	vector <CSUB>       EP;
	vector <int>        poptype;                    // mark as extreme indiv if 1, middle indiv if 0.
	int                 last;

	double              distance;                   // generational distance from PF to solutions found
	int                 popsize;
	void operator=(const CMOEAD &moea);
};

CMOEAD::CMOEAD()
{

	ind_arr = new CMOEADInd[nobj];

	last=0;
	
	// initialize ideal point
	for(int n=0; n<nobj; n++)
	{
		idealpoint.push_back(1.0e+30);
		worstpoint.push_back(-1.0e+30);
		ind_arr[n].rnd_init();
		ind_arr[n].obj_eval();		
	}
	vector <double> INF(nobj, 1.0E+10);
	arch_wp.resize(nobj);

}

CMOEAD::~CMOEAD()
{
	idealpoint.clear();
	worstpoint.clear();
	arch_wp.clear();
	delete [] ind_arr;
}


void CMOEAD::init_population()
{
	for(int i=0; i<population.size(); i++)
	{
		population[i].indiv.rnd_init();
		population[i].indiv.obj_eval();
		if (i < nobj) arch_wp[i] = population[i].indiv;

		update_reference(population[i].indiv);
		nfes++;
	}

	for (int i = 0; i < population.size(); i++)
	{
		update_worstpoint(population[i].indiv);
	}
}

void CMOEAD::operator=(const CMOEAD &alg)
{

	population = alg.population;
	ps         = alg.ps;
	ind_arr    = alg.ind_arr;
	offspring  = alg.offspring;
	distance   = alg.distance;
	popsize    = alg.popsize;
}


// createt the weight vectors with uniform distribution
void CMOEAD::init_uniformweight()
{
	if(nobj==2)
	{
		for(int n=0; n<pops; n++)
		{
			CSUB sub;
			double a = 1.0*n/(pops - 1);
			sub.namda.push_back(a);
			sub.namda.push_back(1-a);
			population.push_back(sub);
		}
		popsize = pops;
	}
	else
	{
		for(int i=0; i<=unit; i++)
		{
			for(int j=0; j<=unit; j++)
			{
				if(i+j<=unit)
				{
					CSUB sub;
					sub.array.push_back(i);
					sub.array.push_back(j);
					sub.array.push_back(unit-i-j);
					for(int k=0; k<sub.array.size(); k++)
						sub.namda.push_back(1.0*sub.array[k]/unit);
					population.push_back(sub);
				}
			}
		}

		popsize = population.size();
		pops    = popsize;
	}
}

// reinitialize weight vectors
void CMOEAD::reinit_uniformweight()
{
	for (int n = 0; n < pops; n++)
	{
		double sum=0;
		for (int j = 0; j < nobj; j++)
		{
			population[n].namda[j] = (1.0 - population[n].namda[j]) / (nobj - 1);
			sum += population[n].namda[j];
		}
		for (int j = 0; j < nobj; j++)
			population[n].namda[j] = population[n].namda[j] / sum;
	}	
	
}

void CMOEAD::init_neighbourhood()
{
    double *x   = new double[population.size()];
	int    *idx = new int[population.size()];
	for(int i=0; i<population.size(); i++)
	{
		// calculate the distances based on weight vectors
		for(int j=0; j<population.size(); j++)
		{
		    x[j]    = dist_vector(population[i].namda,population[j].namda);
			idx[j]  = j;
		}
	
		// find 'niche' nearest neighboring subproblems
		minfastsort(x,idx,population.size(),niche);
		for(int k=0; k<niche; k++)
		{
			population[i].table.push_back(idx[k]);
		}

	}
    delete [] x;
	delete [] idx;
}

//reinitialize neighbourhood
void CMOEAD::reinit_neighbourhood()
{
    double *x   = new double[population.size()];
	int    *idx = new int[population.size()];
	for(int i=0; i<population.size(); i++)
	{
		// calculate the distances based on weight vectors
		for(int j=0; j<population.size(); j++)
		{
		    x[j]    = dist_vector(population[i].namda,population[j].namda);
			idx[j]  = j;
		}
	
		// find 'niche' nearest neighboring subproblems
		minfastsort(x,idx,population.size(),niche);
		for(int k=0; k<niche; k++)
		{
			population[i].table[k]=idx[k];
		}

	}
    delete [] x;
	delete [] idx;
}

// make a match between current population members and subproblems
void CMOEAD::population_match()
{
	vector<CSUB> curpop(population);

	double min, tmp;
	int mark;
	for (int i = 0; i < pops; i++)
	{
		mark = i;
		min = 1.0e+30;
		for (int j = 0; j < pops; j++)
		{
			tmp = fitnessfunction(curpop[j].indiv.y_obj, population[i].namda, ind_arr);
			if (tmp < min)
			{
				min = tmp;
				mark = j;
			}
		}
		population[i].indiv = curpop[mark].indiv;
	}
	curpop.clear();
}

// update solution to each subproblem
void CMOEAD::update_problem(CMOEADInd &indiv, int id, int type, int rule)
{
	// indiv: child solution
	// id:   the id of current subproblem
	// type: update solutions in - neighborhood (1) or whole population (otherwise)
	int size, time = 0;
	if(type==1)	size = population[id].table.size();
	else        size = population.size();
	int *perm = new int[size];
	random_permutation(perm, size);

	if (rule == 0)
	{
		for (int i = 0; i < size; i++)
		{
			int k;
			if (type == 1) k = population[id].table[perm[i]];
			else        k = perm[i];

			// calculate the values of objective function regarding the current subproblem
			double f1, f2;

			f1 = fitnessfunction(population[k].indiv.y_obj, population[k].namda, ind_arr);
			f2 = fitnessfunction(indiv.y_obj, population[k].namda, ind_arr);

			if (otype)
			{
				if (f1 < f2)
				{
					population[k].indiv = indiv;
					time++;
				}
			}
			else
			{
				if (f2 < f1)
				{
					population[k].indiv = indiv;
					time++;
				}
			}

			// the maximal number of solutions updated is not allowed to exceed 'limit'
			if (time >= limit)
			{
				return;
			}

		}
	}

	else
	{
		double f1, f2;
		f1 = fitnessfunction(population[id].indiv.y_obj, population[id].namda, ind_arr);
		f2 = fitnessfunction(indiv.y_obj, population[id].namda, ind_arr);

		if ((otype == 1))
		{
			if (f1<f2)
			{
				population[id].indiv = indiv;
			}
		}
		else
		{
			if (f2<f1)
			{
				population[id].indiv = indiv;
			}
		}
	}
	delete [] perm;
}

// Attainable Scalarizing function
double  CMOEAD::ASF_function(vector<double> vect,int n)
{
	double max = -1.0e10;
	double w, tmp;
	for (int j = 0; j < nobj; j++)
	{
		w = EPS;
		if (j == n)
			w = 1.0;
		tmp = (vect[j] - idealpoint[j]) / w;
		if (tmp > max)
			max = tmp;
	}

	return max;
}

// update reference/idea point
void CMOEAD::update_reference(CMOEADInd &ind)
{
	//ind: child solution
	for(int n=0; n<nobj; n++)
	{

		if((ind.y_obj[n]<idealpoint[n]&&otype==0))
		{
			
			idealpoint[n] = ind.y_obj[n];
			ind_arr[n]    = ind;
		}

	}	
}

// update the worst objectives of the population
void CMOEAD::update_worstpoint(CMOEADInd &ind)
{

	double asf1, asf2;
	for (int n = 0; n < nobj; n++)
	{
		bool better = true;
		asf1 = ASF_function(ind.y_obj, n);
		for (int i = 0; i < nobj; i++)
		{
			asf2 = ASF_function(arch_wp[i].y_obj, n);
			if (asf1 > asf2)
			{
				better = false;
				break;
			}
		}
		if (better) arch_wp[n] = ind;
	}

	for (int n = 0; n < nobj; n++)
	{
		worstpoint[n] = -1.0e+30;
		for (int i = 0; i < nobj; i++)
		{
			if (worstpoint[n] < arch_wp[i].y_obj[n])
				worstpoint[n] = arch_wp[i].y_obj[n];
		}
	}
}

// calculate niche count
double CMOEAD::nichecount(int cid, int flag)
{
	int ss=population[cid].table.size();
	double nc=0; //niche_count
	double d=0;
	for (int i=0;i<ss;i++)
	{
		int j=population[cid].table[i];
		d=dist_vector(population[cid].indiv.y_obj,population[j].indiv.y_obj);
		if (flag==1)
			if (d<sh)
				nc+=1-pow(d/sh,2);
			else
				nc+=0;
		else
			nc+=d;
	}
	return nc;
}

// two-phase subroutine
void CMOEAD::TP_response(bool flag)
{
	if (!flag) return;

	double dmid = 0, detr = 0;
	int nmid = 0, netr = 0;

	for (int i = 0; i < population.size(); i++)
	{
		double prodw = 1;
		for (int j = 0; j < nobj; j++)
		{
			prodw *= population[i].namda[j];
		}
		if (prodw < 0.5*pow(1.0 / nobj, nobj))
		{

			detr += nichecount(i, 0);
			netr += 1;
		}
		else
		{
			dmid += nichecount(i, 0);
			nmid += 1;
		}
		EP.push_back(population[i]);

	}

	dmid /= nmid, detr /= netr;

	if (dmid < detr)
	{
		otype = 1;
		reinit_uniformweight();
		reinit_neighbourhood();

	}

	//-------the following as an option will make the algorithm work more efficiently by 
	//-------establishing a match between current population and the new weight vectors
	population_match();
}

// original mating selection method
void CMOEAD::matingselection1(vector<int> &list, int cid, int size, int type){
	// list : the set of the indexes of selected mating parents
	// cid  : the id of current subproblem
	// size : the number of selected mating parents
	// type : 1 - neighborhood; otherwise - whole population
	int ss   = population[cid].table.size(), r, p;
    while(list.size()<size)
	{
		if(type==1){
		    r = int(ss*rnd_uni(&rnd_uni_init));
			p = population[cid].table[r];
		}
		else
			p = int(population.size()*rnd_uni(&rnd_uni_init));

		bool flag = true;
		for(int i=0; i<list.size(); i++)
		{
			if(list[i]==p) // p is in the list
			{ 
				flag = false;
				break;
			}
		}

		if(flag) list.push_back(p);
	}
}

// niche-guided mating selection method
void CMOEAD::matingselection2(vector<int> &list, int cid, int size, int rule){
	// list : the set of the indexes of selected mating parents
	// cid  : the id of current subproblem
	// size : the number of selected mating parents
	// type : 1 - neighborhood; otherwise - whole population
	int ss   = population[cid].table.size(), r, p;
    while(list.size()<size)
	{
		if(rule==1){
			p = int(population.size()*rnd_uni(&rnd_uni_init));
			for (r=0;r<ss;r++)
				if (p==population[cid].table[r])
					break;
		}

		if (r<ss)
			continue;

		bool flag = true;
		for(int i=0; i<list.size(); i++)
		{
			if(list[i]==p) // p is in the list
			{ 
				flag = false;
				break;
			}
		}

		if(flag) list.push_back(p);
	}
}

void CMOEAD::diffevolution()
{
	pops = population.size();
	int *perm = new int[pops];
	random_permutation(perm, pops);

    for(int i=0; i<pops; i++)
	{
		int n = perm[i];

		// calculate niche count
		double nc=nichecount(n,1);
		int rule=0;

		if ((nc>1.0*niche / 2) && (rnd_uni(&rnd_uni_init) > 0.5))
		{
			rule = 1;
		}
		int type;
        double rnd = rnd_uni(&rnd_uni_init);

		// mating selection based on probability
		if(rnd<realb)    type = 1;   // neighborhood
		else             type = 2;   // whole population

		// select the indexes of mating parents
		vector<int> p;
		if (rule==0)
			matingselection1(p,n,2,type);  // original neighborhood selection
		else
			matingselection2(p,n,2,rule); //niche-guided selection

		// produce a child solution
		CMOEADInd child;
		diff_evo_xover2(population[n].indiv,population[p[0]].indiv,population[p[1]].indiv,child);

		// apply polynomial mutation
		realmutation(child, 1.0/nvar);
		
		// evaluate the child solution
		child.obj_eval();

		// update the reference points and other solutions in the neighborhood or the whole population
		update_reference(child);
		update_worstpoint(child);

		update_problem(child, n, type, rule);


		p.clear(); 	nfes++; 
	}

    delete [] perm;
}


void CMOEAD::execute(int run, char *strfunc, char *stralg)
{
	seed = (seed + 23)%1377;
	rnd_uni_init = -(long)seed;

    strcpy_s(strFunctionType,strfunc);
	strcpy_s(strAlgorithmType,stralg);


    char filename[1024];
	char fileparam[1024];
	

	// initialization 
	int gen   = 1;
	nfes      = 0;
	otype=0;
	init_uniformweight();
    init_neighbourhood();
	init_population();

	// evolution
	for(gen=1; gen<=max_gen; gen++)
	{
		itt=gen;
		
		diffevolution();
		cout<<"gen = "<<gen<<endl;

		int Tg=(int) (Gr*max_gen);
		TP_response(gen == Tg);

       if(gen==max_gen)
		{
		   sprintf_s(filename,"%s\\pop_gen_%i.txt",parName,gen);
	       save_front(filename);
		}
	}
	//-------truncation of final nondominated set--------// 
	/*nondominate_truncation();
	population.clear();
	sprintf_s(filename, "%s\\lastpop.txt", parName);
	save_front(filename);*/
	//---------------end of truncation------------------//
	EP.clear();
}



// identification and truncation of nondominated set of population and archive 
void CMOEAD::nondominate_truncation()
{
	
	EP.reserve(2*population.size());
	EP.insert(EP.end(), population.begin(), population.end());

	// remove duplicate members
	for (int i = 0; i < EP.size(); i++)
	{
		int j = i + 1;
		while(j<EP.size()){
			if (EP[i].indiv == EP[j].indiv)
				EP.erase(EP.begin()+j);
			else j++;
		}
	}

	nondominate_rank_pop();
	if (EP.size()>popsize)
		farthest_first_selection();
		//kth_neighbor_selection();
}

// population ranking : identification of nondominated solutions
void CMOEAD::nondominate_rank_pop()
{
	// rank: only identify nondominated solutions in the population (0==nondominated)

	for (int i = 0; i <EP.size(); i++)
	{
		EP[i].indiv.rank = 0;
	}

	int dominated;
	for (int i = 0; i < EP.size(); i++)
	{
		for (int j = 0; j < EP.size(); j++)
		{
			dominated = dominance(EP[i].indiv.y_obj, EP[j].indiv.y_obj);
			if (dominated == -1) EP[i].indiv.rank++;
		}
	}

	int i = 0;
	while (i<EP.size())
	{
		//if (EP.size() == popsize) return;
		if (EP[i].indiv.rank>0)
			EP.erase(EP.begin() + i);
		else i++;

	}
}

// the-farthest-the-first method for solution truncation
void CMOEAD::farthest_first_selection()
{
	double min, max;
	int size = EP.size();

	int id1, id2, tmp;

	vector <int> accept, v2;
	for (int i = 0; i < size; i++)
	{
		v2.push_back(i);
		EP[i].indiv.rank = 0;
	}


	//  find the extreme point on the y_obj[0] objective
	min = 1.0e+30, max = -1.0e+30;
	for (int i = 0; i<size - accept.size(); i++)
	{
		if (min>EP[i].indiv.y_obj[0]) id1 = i;
		if (max < EP[i].indiv.y_obj[0]) id2 = i;
	}
	tmp = v2[id1];
	accept.push_back(v2[id1]); EP[v2[id1]].indiv.rank = 1;
	v2[id1] = v2[size - accept.size()];
	v2[size - accept.size()] = tmp;

	if (id1 != id2)
	{
		tmp = v2[id2];
		accept.push_back(v2[id2]); EP[v2[id2]].indiv.rank = 1;
		v2[id2] = v2[size - accept.size()];
		v2[size - accept.size()] = tmp;
	}


	double *dist = new double[v2.size() - accept.size()];

	for (int i = 0; i < v2.size() - accept.size(); i++)
	{
		dist[i] = 1.0e+30;
		for (int j = 0; j<accept.size(); j++)
		{
			min = dist_vector(EP[v2[i]].indiv.y_obj, EP[accept[j]].indiv.y_obj);
			if (dist[i]>min) dist[i] = min;
		}
	}

	// preserve half of population using the-farthest-the-first method 
	while (accept.size() < popsize)
	{
		max = -1.0e+30;
		for (int j = 0; j < v2.size() - accept.size(); j++)
		{
			if (max < dist[j])
			{
				max = dist[j];
				id2 = j;
			}
		}
		tmp = v2[id2];
		accept.push_back(v2[id2]); EP[v2[id2]].indiv.rank = 1;
		v2[id2] = v2[v2.size() - accept.size()];
		v2[v2.size() - accept.size()] = tmp;
		dist[id2] = dist[v2.size() - accept.size()];
		for (int j = 0; j<v2.size() - accept.size(); j++)
		{
			min = dist_vector(EP[v2[j]].indiv.y_obj, EP[tmp].indiv.y_obj);
			if (dist[j]>min) dist[j] = min;
		}

	}

	int i = 0; 
	while(i<EP.size())
	{
		if (!EP[i].indiv.rank) EP.erase(EP.begin() + i);
		else i++;
	}

	delete[] dist;
}

void CMOEAD::save_front(char saveFilename[1024])
{
    std::fstream fout;
	fout.open(saveFilename,std::ios::out);

	for(int n=0; n<population.size(); n++)
	{
		for(int k=0;k<nobj;k++)
			fout<<population[n].indiv.y_obj[k]<<"  ";
		fout<<"\n";
	}
	for (int n=0;n<EP.size();n++)
	{
		for(int k=0;k<nobj;k++)
			fout<<EP[n].indiv.y_obj[k]<<"  ";
		fout<<"\n";
	}
	fout.close();
}

void CMOEAD::save_ps(char saveFilename[1024])
{
    std::fstream fout;
	fout.open(saveFilename,std::ios::out);
	for(int n=0; n<popsize; n++)
	{
		for(int k=0;k<nvar;k++)
			fout<<population[n].indiv.x_var[k]<<"  ";
		fout<<"\n";
	}
	fout.close();
}


void CMOEAD::calc_distance()
{
    distance = 0;
	for(int i=0; i<ps.size(); i++)
	{
	    double min_d = 1.0e+10;
		for(int j=0; j<population.size(); j++)
		{
            double d = dist_vector(ps[i].y_obj, population[j].indiv.y_obj);
			if(d<min_d)  min_d = d;
		}
		distance+= min_d;
	}
	distance/=ps.size();
}

#endif