/**************************************************************************************/
/* GA4DOP.c                                                                           */
/* This is an implementation of memory enhanced genetic algorithms for dynamic        */
/* optimisation problems with the XOR generator for the following paper:              */
/*                                                                                    */
/*   S. Yang and X. Yao. Population-based incremental learning with associative       */
/*   memory for dynamic environments. IEEE Transactions on Evolutionary Computation,  */
/*   12(5): 542-561, October 2008. IEEE Press (DOI: 10.1109/TEVC.2007.913070).        */
/*                                                                                    */
/* Compile:                                                                           */
/*   g++ -o GA4DOP GA4DOP.c                                                           */
/* Run:                                                                               */
/*   ./GA4DOP gaMode funcNo chgMode chgDeg chgSpeed immRate rSeed                     */
/*   ./GA4DOP gaMode funcNo chgMode chgDeg chgSpeed immRate pBaseMut rSeed            */
/*                                                                                    */
/* Written by:                                                                        */
/*   Shengxiang Yang, University of Leicester, UK, May 2005; refined November 2009    */
/*                                                                                    */
/* If any query, please email Shengxiang Yang at s.yang@mcs.le.ac.uk                  */
/*                                                                                    */
/**************************************************************************************/

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string>
#include <fstream>

using namespace std;

/* Change any of these parameters to match your needs */

#define NGENES 100    // no. of genes in a binary chromosome
#define POPSIZE 100   // population size
#define MEMSIZE 10    // memory population size
#define MAXBASESTATES 20  // total base states for cyclical changes

/* Global variables */
FILE *logPopBest;     // output file for best-of-pop fitness over generations
FILE *logPopDiv;      // output file for pop diversity
int currentGenNo=0;   // current generation no

double pMutation = 0.02;    // prob. of mutation
double pCrossover = 0.6;    // prob. of crossover
double pParamUX = 0.5;      // swapping probability for parameterized UX
double immigrateRatio =0.2; // ratio of random immigrants to total population size
int nImmigrants;            // number of random immigrants

int randSeed = 0;
int gaMode = 1;   // GA mode (with the elitism of size 1): 
                  //   1-MEGA2r (memory-enhanced GA with two pops and restart),  
                  //   2-ISGA with best memory point and dynamic gene pool and 
                  //     aligned transformation

int funcNo = 1;    // NK Function: 1-NK(25,4,1), 2-NK(25,4,3), 3-NK(25,4,4), 
                    //  4-NK(25,4,deceptive), 25 copies of 4-bit deceptive problems
int changeMode = 1; // how environment changes, cyclical or not
                    //   1: cyclic, 2: cyclic with noise, 3: random
double changeDegree = 0.5; // the ratio of loci changed in dynamic environment
int changeSpeed = 10;      // the period of environment changing in generations
int totalChanges = 200;     // the total number of environment changes in a run
bool changeDetected;       // whether the enviroment change has been detected

int xorMask[NGENES];    // the mask for XORing environment
bool bRandRatio =false; // whether the changeDegree is random (default to false)
int baseXORMasks[MAXBASESTATES][NGENES]; // base xor masks for cyclical environment
int totalBaseStates;
double pBaseMut = 0.05; // probability of mutating each element of a base XOR mask 

int popSize1;     // the size of 1st population
int popSize2;     // the size of 2nd population
int newPopSize1;  // the size of 1st population in next generation
int newPopSize2;  // the size of 2nd population in next generation
int minPopSize1;  // the min size of 1st population
int minPopSize2;  // the min size of 2nd population
int maxPopSize1;  // the max size of 1st population
int maxPopSize2;  // the max size of 2nd population
int deltaPopSize; // the amount of pop size to be adjusted for two populations

int memSize;           // the size of memory
int memLevel;          // number of points in the memory (full if memLevel == memSize) 
int nextMemUpdateTime; // next memory update generation for MGA, memory updated after 
                       // next random number of gnerations in [5, 10]

struct genotype {      // genotype (GT), a member of the population
  int gene[NGENES];       
  int phenoGene[NGENES];  // phenotype allele for dynamic environment

  double fitness;         // phenotype fitness
  double rfitness;        // relative fitness
  double sfitness;        // scaled fitness
  double cfitness;        // cumulative fitness
  double oldfitness;      // fitness of previous generation for memory points
                          // used for detecting environmental changes 
};

struct genotype pop[POPSIZE+1];     // population
struct genotype newpop[POPSIZE+1];  // temporary population
struct genotype tmpMember;          // temporary individual for sortPopulation()

#define numSegs 200
#define maxSegLength 50
#define minSegLength 5
#define fixSegLength 5
int geneSegPool[numSegs][fixSegLength]; // the gene segment pool
int geneSegPosition[numSegs];            // the starting locus for each gene segment
int geneSegLength[numSegs];              // the length for each gene segment
bool bRandomSeg[numSegs];                // whether a segment is random or not
double pTransform = 0.9;                 // transformation rate for ISGAs
double replaceRate = 0.2;                // percentage of gene segments to update   
double randomSegRatio = 0.3;             // percentage of random gene segments
int randomSegStartIdx;                  
int totalReplaces; 

/***************************************************/
/* Utility Functions                               */
/***************************************************/
int randomBit() { return (rand()%2); }

double randNumber(double low, double high) {
  return ((double)(rand()%10000)/10000.0)*(high - low) + low;  
}

int sign(double value) {
  if (value == 0.0) return 0; 
  else return (value > 0.0? 1 : -1);
}

double abs(double value) {
  return (value > 0.0? value: (-1.0*value));
}

template <typename T>
void swap(T *x, T *y) {
  T temp;
  temp = *x; *x = *y; *y = temp;
}

/***********************************************************/
/* Random sequence generator:                              */
/*   Generates a random sequence of seqSize numbers from   */
/*   the natural number array [0, 1, ..., numSize-1]       */
/***********************************************************/
void randomSequence(int* seqArr, int seqSize, int numSize) {
  int i, j, idx, count, numleft;

  // constructs a natural number array
  int number[numSize];
  for (i = 0; i < numSize; ++i) number[i] = i; 

  // select seqSize numbers from number[] without repeat
  numleft = numSize;
  for (i = 0; i < seqSize; ++i) { 
    idx = rand() % numleft;
    count = 0;
    for (j = 0; j < numSize; ++j) { 
      if (number[j] != -1) // found one not selected number
        ++count;

      if (count > idx) break; // found the idx-th not selected number
    }

    seqArr[i] = number[j];
    number[j] = -1; // marked as selected
    --numleft;
  }
}

/******************************************************/
/* DUF function                                       */
/******************************************************/
#define NBLOCKS 25    // the number of schemas
#define BLOCKSIZE 4   // block size - length of target schema

void DUF(int mem) {
  int i, j, countOFOnes;

  for (i = 0; i < NGENES; ++i) // perform XORing operation
    pop[mem].phenoGene[i] = pop[mem].gene[i] ^ xorMask[i];

  pop[mem].fitness = 0.0;
  for (i = 0; i < NBLOCKS; ++i) { 
    countOFOnes=0;
    for (j = 0; j < BLOCKSIZE; j++)
      if (pop[mem].phenoGene[i*BLOCKSIZE+j] == 1)
        countOFOnes++;  // count the 1 bits in the block

    switch(funcNo) {
      case 1: // DUF(25,4,1)
        pop[mem].fitness += countOFOnes; 
        break;
      case 2: // DUF(25,4,2)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else if(countOFOnes == 3) pop[mem].fitness += 2;
        break;
      case 3: // DUF(25,4,4)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        break;
      case 4: // DUF(25,4,deceptive)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else pop[mem].fitness += (3 - countOFOnes);
        break;
    }
  }
}

/***************************************************************/
/* Initialization function: Initializes the values of genes    */
/* in binary. It also initializes (to zero) all fitness values */
/* for each member of the main and memory populations. It      */
/* initializes the percentage of 1's to be 50% for each gene   */
/* locus.                                                      */
/***************************************************************/
void initialize() {
  int i, j;

  // intitialize the mask to static problem
  for (i=0; i<NGENES; ++i) xorMask[i] = 0;

  switch(gaMode) {
    case 1:  // MEGA2r
      popSize1 = (int)(0.45*POPSIZE);
      popSize2 = (int)(0.45*POPSIZE);
      newPopSize1 = popSize1;
      newPopSize2 = popSize2;
      memSize = (int)(0.1*POPSIZE);
      minPopSize1 = (int)(0.3*POPSIZE);
      minPopSize2 = (int)(0.3*POPSIZE);
      maxPopSize1 = (int)(0.6*POPSIZE);
      maxPopSize2 = (int)(0.6*POPSIZE);
      deltaPopSize = (int)(0.05*POPSIZE); 
      memLevel = 0;
      break;
    case 2: // ISGA
      popSize1 = (int)(0.9*POPSIZE);
      newPopSize1 = popSize1;
      memSize = (int)(0.1*POPSIZE);
      memLevel = 0;
      break;
  }

  // initialize the whole population including memory if used
  for (i = 0; i <= POPSIZE; ++i) 
    for (j = 0; j < NGENES; ++j) {
        pop[i].gene[j] = randomBit();
        pop[i].phenoGene[j] = pop[i].gene[j];
    }

  for (i = 0; i <= POPSIZE; ++i) {
    pop[i].fitness = 0.0;
    pop[i].rfitness = 0.0;
    pop[i].sfitness = 0.0;
    pop[i].cfitness = 0.0;
    pop[i].oldfitness = 0.0;
  }

  if (gaMode == 2) { // ISGA

    // initialize gene segments randomly
    for (i=0; i<numSegs; ++i) {
      geneSegLength[i] = fixSegLength;
      geneSegPosition[i] = rand()%(NGENES-fixSegLength);

      for (j=0; j<fixSegLength; ++j) 
        geneSegPool[i][j] = randomBit();
    }

    randomSegStartIdx = (int)((1.0 - randomSegRatio)*numSegs);

    replaceRate = 0.2;
    totalReplaces = (int)(replaceRate*numSegs);

    for (i=0; i<randomSegStartIdx; ++i)
      bRandomSeg[i] = false;
    for (i=randomSegStartIdx; i<numSegs; ++i)
      bRandomSeg[i] = true;
  }

  nImmigrants = (int)(immigrateRatio * POPSIZE);
  changeDetected = false;
}

/***********************************************************/
/* Sort sub-population in the order of decreasing fitness  */
/*   using the bubble-sorting algorithm                    */
/***********************************************************/
void sortPopulation(int start, int end) {
  int i, j;
  for (i = start; i < end; ++i) 
    for (j = start + 1; j < start + end - i; ++j)
      if (pop[j-1].fitness < pop[j].fitness) {
        tmpMember = pop[j-1];
        pop[j-1] = pop[j];
        pop[j] = tmpMember;
      }
}

/***********************************************************/
/* evaluate the main population, excluding memory if used  */
/***********************************************************/
void evaluatePop() {
  for (int mem = 0; mem < POPSIZE - memSize; ++mem) 
    DUF(mem);
}

/***********************************************************/
/* evaluate memory and detect environmental changes        */
/***********************************************************/
void evaluateMemory() {
  int i, memStartID = POPSIZE - memSize;
  for (i = memStartID; i < POPSIZE; ++i) // store old fitness
    pop[i].oldfitness = pop[i].fitness;

  changeDetected = false; 
  for (i = memStartID; i < POPSIZE; ++i) {
    DUF(i);

    // if not the initial generation, check whether change occurs
    if (currentGenNo != 0) {  
      if (pop[i].oldfitness != pop[i].fitness)
        changeDetected = true;
    }
  }
}

void storeElite() {
  int idx;
  switch(gaMode) {
    case 1: 
      sortPopulation(0, popSize1);  // sort population 1 
      sortPopulation(popSize1, popSize1+popSize2);  // sort population 2
      idx = (pop[0].fitness > pop[popSize1].fitness) ? 0 : popSize1;
      break;
    case 2: 
      sortPopulation(0, popSize1);  // sort population
      idx = 0;
      break;
    default: break;
  }

  pop[POPSIZE] = pop[idx]; // store elite
}

// replace the worst individual in pop 1 with stored elite
void retrieveElite() {
  sortPopulation(0, popSize1);  // sort population 1
  pop[popSize1-1] = pop[POPSIZE];

  DUF(popSize1-1); // re-evaluate the elite in case change occurs
}

/**************************************************************/
/* Adjust pop sizes: Adjust sizes for pops according to their */
/*     performance. Winner wins deltaPopSize from loser.      */
/**************************************************************/
void adjustPopSize() {
  double bestFitness1 = pop[0].fitness;
  double bestFitness2 = pop[popSize1].fitness;

  for (int i = 1; i < popSize1; ++i)
    if (bestFitness1 < pop[i].fitness)
      bestFitness1 = pop[i].fitness;

  for (int i = popSize1+1; i < popSize1+popSize2; ++i) 
    if (bestFitness2 < pop[i].fitness)
      bestFitness2 = pop[i].fitness;

  if ( bestFitness1 > bestFitness2 )
    newPopSize1 = min(popSize1 + deltaPopSize, maxPopSize1);
  else if ( bestFitness1 < bestFitness2 )
    newPopSize1 = max(popSize1 - deltaPopSize, minPopSize1);
  newPopSize2 = POPSIZE - newPopSize1 - memSize;
}

/***************************************************************/
/* calcCumulateFitness: calculate the cumulative fitness of a  */
/*   pop using the sigma truncation scaling method descibed in */
/*   goldberg p 124. If the scaled fitness is less than zero,  */
/*   it is arbitrarily set to zero (thus the truncation part   */
/*   of 'sigma truncation').                                   */
/***************************************************************/
void calcCumulateFitness(int start, int end) {
  // find total fitness of the population      
  double sum = 0.0;
  for (int i = start; i < end; ++i)
    sum += pop[i].fitness;

  // calculate relative fitness
  if (sum == 0.0) { // all members have 0 fitness
    for (int i = start; i < end; ++i)
      pop[i].rfitness = 1.0/(end - start);  // set relative fitness
  }
  else {
    for (int i = start; i < end; ++i) // calculate relative fitness
      // pop[i].rfitness = pop[i].sfitness/sum;
      pop[i].rfitness = pop[i].fitness/sum;
  }

  // calculate cumulative fitness
  pop[start].cfitness = pop[start].rfitness;
  for (int i = start + 1; i < end; ++i)
    pop[i].cfitness =  pop[i-1].cfitness + pop[i].rfitness;
}

/**************************************************************/
/* Selection function: Standard proportional selection for    */
/* maximization problems                                      */
/**************************************************************/
void select() {
  int i, j, mem, index;
  double ptr, p, sum = 0.0;

  // adjusts next pop sizes for two pops
  if (gaMode==1) adjustPopSize();

  /**********************************************************  
   * Using standard fitness proportional selection without  *
   *    Sigma Trunction Scaling                             *
   **********************************************************/
  // calculate cumulative fitness for pop 1
  calcCumulateFitness(0, popSize1);

  // select newPopSize1 survivors using cumulative fitness.
  for (i = 0; i < newPopSize1; ++i) { 
    p = randNumber(0.0, 1.0);
    if (p < pop[0].cfitness) newpop[i] = pop[0];      
    else {
      for (j = 0; j < popSize1; ++j)
        if (p >= pop[j].cfitness && p<pop[j+1].cfitness)
          { newpop[i] = pop[j+1]; break; }
    }
  }

  if (gaMode == 1) {
    // calculate cumulative fitness for pop 2
    calcCumulateFitness(popSize1, popSize1+popSize2);

    // select newPopSize2 survivors using cumulative fitness.
    for (i = newPopSize1; i < POPSIZE - memSize; ++i) { 
      p = randNumber(0.0, 1.0);
      if (p < pop[popSize1].cfitness) newpop[i] = pop[popSize1];      
      else {
        for (j = popSize1; j < POPSIZE - memSize; ++j)
          if (p >= pop[j].cfitness && p<pop[j+1].cfitness)
            { newpop[i] = pop[j+1]; break; }
      }
    }

    // update pop size variables
    popSize1 = newPopSize1;
    popSize2 = newPopSize2;
  }

  /* once a new population is created, copy it back */
  for (i = 0; i < POPSIZE - memSize; ++i) 
    pop[i] = newpop[i];
}

/**************************************************************/
/* Uniform Crossover: Parameterized                           */
/**************************************************************/
void UniformXover(int one, int two) {
  if(NGENES > 2) {
    for (int i = 0; i < NGENES; ++i)
      if ( randNumber(0.0, 1.0) < pParamUX ) { // pParamUX = 0.5
        if (pop[one].gene[i] != pop[two].gene[i])
          swap(pop[one].gene[i], pop[two].gene[i]);
      }
  }
}

/***************************************************************/
/* Crossover selection: selects two parents that take part in  */
/* the crossover. Implements uniform crossover                */
/***************************************************************/
void crossover(int start, int end) {
  int i, j, k;
  int psize = end - start;

  /* shuffle the population */
  int index[psize];  // random index array
  randomSequence(index, psize, psize);

  for (i=0; i<psize; ++i) newpop[start+i] = pop[start+i];

  for (i=0; i<psize; ++i) pop[start+i] = newpop[start+index[i]];

  int mem, one;
  int first = 0;   // count of the number of members chosen
  double p;
  for (mem = start; mem < end; ++mem) {
    p = randNumber(0.0, 1.0);
    if (p < pCrossover) {
      ++first;
      if (first%2 == 0) UniformXover(one, mem); 
      else one = mem;
    }
  }
}

/**************************************************************/
/* Update Gene Pool: update gene segments from the pop member */
/*   randomly                                                 */
/**************************************************************/
void updateGenePool(void) {
  int i, j, one, two, winner, length, pos;

  // set the random gene segments
  for (i=randomSegStartIdx; i<numSegs; ++i) {
    // length = rand()%(maxSegLength-minSegLength+1) + minSegLength;
    // geneSegLength[i] = length;
    // geneSegPosition[i] = rand()%(NGENES-length);
    // for (j=0; j<length; ++j) 
    //   geneSegPool[i][j] = randomBit();

    geneSegPosition[i] = rand()%(NGENES-fixSegLength);
    for (j=0; j<fixSegLength; ++j) 
      geneSegPool[i][j] = randomBit();
  }
  
  // update totalReplaces non-random gene segments from the current population
  int index[totalReplaces];
  randomSequence(index,totalReplaces,randomSegStartIdx);

  for (i=0; i<totalReplaces; ++i) {
    // tournament select one member from which to update a gene segment    
    one = rand()%popSize1;       // one in [0, popSize1-1]
    two = rand()%(popSize1 - 1); // two in [0, popSize1-2]
    if (two >= one) two++;
    winner = (pop[one].fitness > pop[two].fitness ? one : two);

    // length = rand()%(maxSegLength-minSegLength+1) + minSegLength;
    // geneSegLength[index[i]] = length; 
    // pos = rand()%(NGENES-length);
    pos = rand()%(NGENES-fixSegLength);
    geneSegPosition[index[i]] = pos; 

    for (j=0; j<fixSegLength; ++j) 
      geneSegPool[index[i]][j] = pop[winner].gene[pos+j];
  }
}

/*************************************************************/
/* Transformation: select one gene segment from the library  */
/*   randomly and transform it into a member from a random   */  
/*   position                                                */
/*************************************************************/
void transform(int mem) {
  int i, j, stop, segIdx, pos;

  if (randNumber(0.0, 1.0) < pTransform) {
    segIdx = rand()%numSegs;

    if (bRandomSeg[segIdx] == true) 
      pos = rand()%NGENES;
    else // aligned transformation
      pos = geneSegPosition[segIdx];

    stop = min(NGENES, pos+geneSegLength[segIdx]);
    for (i=pos,j=0; i<stop; ++i,++j) 
      pop[mem].gene[i] = geneSegPool[segIdx][j];
  }
}

/*****************************************************/
/* Clone: clone the population                       */
/*****************************************************/
void clone(void) {
  for (int i=0; i<POPSIZE -memSize; ++i)
    transform(i);
}

/**************************************************************/
/* Mutation: Random uniform mutation. A bit selected for      */
/* mutation is bit flipped                                    */
/**************************************************************/
void mutate(double pmut) {
  double p;
  for (int i = 0; i < POPSIZE - memSize; ++i) {
    for (int j = 0; j < NGENES; ++j) {
      p = randNumber(0.0, 1.0);
      if( p < pmut )
        pop[i].gene[j] = ((pop[i].gene[j]==0)? 1 : 0);
    }
  }
}

/*********************************************************/
/* Restart pop 2 when environment change is deteced      */
/*********************************************************/
void restart() {
  if (gaMode == 1) { // initialize the 2nd population
    for (int mem = popSize1; mem < popSize1+popSize2; ++mem) {
      for (int i = 0; i < NGENES; ++i) {
        pop[mem].gene[i] = randomBit();
        pop[mem].phenoGene[i] = pop[mem].gene[i];
      }

      DUF(mem); // evaluate the fitness
      pop[mem].oldfitness = 0.0;
    }
  }
}

/*****************************************************/
/* Retrieve memory: merge memory with main pop and   */
/* select best individuals as the new main pop       */
/*****************************************************/
void retrieveMemory() {
  int i, j;
  if (gaMode == 1) {
    for (i = 0; i < POPSIZE; ++i)   // copy pop + memory
      newpop[i] = pop[i];

    if (POPSIZE > popSize1 + memSize) {  // 2 pops
      j = popSize1;
      for (i=POPSIZE - memSize; i<POPSIZE; ++i, ++j)  // shift memory upward
        pop[j] = pop[i];
    }

    sortPopulation(0, popSize1 + memSize);  // sort pop + memory points 

    // select popSize1 best individuals as the new pop 1
    for (i = 0; i < popSize1; ++i) 
      newpop[i] = pop[i];

    // copy the whole pop back without changing the memory
    for (i = 0; i < POPSIZE; ++i)
      pop[i] = newpop[i];
  }
  else { // ISGA
    if (memLevel != 0) {
      int memIdx = POPSIZE - memSize; // index of memory point to retrieve
 
      sortPopulation(memIdx, memIdx+memLevel);  // sort memory

      // clone from the most closest or best memory point into population
      sortPopulation(0, popSize1);   // sort pop 1 
      pop[popSize1-1] = pop[memIdx]; // keep the best of memory
      for (i = 1; i < nImmigrants; ++i) {
        for (j = 0; j < NGENES; ++j) 
          pop[popSize1-1-i].gene[j] = pop[memIdx].gene[j];
        transform(popSize1-1-i);
      }
    }
    else {  // random immigrate
      for (i=0; i<nImmigrants; ++i) {
        for (j=0; j<NGENES; ++j)
          pop[popSize1-1-i].gene[j] = randomBit();
        transform(popSize1-1-i);
      }
    }
  }
}

/******************************************************/
/* Retrieve best individual from main population to   */
/* replace closest memory point                       */
/******************************************************/
void updateMemory(void) {
  int i, j;
  int bestindex = popSize1; // index of the best individual of population
  int memindex;             // index of the memory point to be changed
  int tmpDist, minHammingDist = NGENES;
  int memStartIdx = POPSIZE - memSize;

  switch(gaMode) {
    case 1:
      sortPopulation(0, popSize1);
      sortPopulation(popSize1, popSize1+popSize2);
      bestindex = (pop[0].fitness >= pop[popSize1].fitness)? 0 : popSize1;
      break; 
    case 2:
      sortPopulation(0, popSize1);
      bestindex = 0;
      break;
    default: break;
  }

  // if memory not full, store as new point; otherwise, replace the closest
  if (memLevel < memSize) {
    pop[memStartIdx + memLevel] = pop[bestindex];
    memLevel++;
  }
  else { // memory is full
    // find the memory point closest to the best individual of pop
    memindex = memStartIdx;
    for (i = memStartIdx; i < POPSIZE; ++i) {
      tmpDist = 0;
      for (j = 0; j < NGENES; ++j) { // calculate edit distance
        if (pop[i].gene[j] != pop[bestindex].gene[j]) 
          tmpDist++;
      }

      if (tmpDist < minHammingDist) {
        minHammingDist = tmpDist; 
        memindex = i;
      }  
    }

    // update memory point if the best individual of pop is better
    if (pop[memindex].fitness < pop[bestindex].fitness) {
      pop[memindex] = pop[bestindex];
    }
  }
}

/*********************************************************/
/* Environment Dynamics Function: Construct base XORing  */
/* masks for (partially) cyclical changing environments  */
/*********************************************************/
void constructBaseXORMasks(void) {
  int i, j, numOnes;
  int randIndex[NGENES];

  // create a random permutation of 0 to NGENES-1
  randomSequence(randIndex, NGENES, NGENES);

  // initialize the baseXORMasks
  for (i=0; i<totalBaseStates; ++i)
    for (j=0; j<NGENES; ++j) baseXORMasks[i][j] = 0;

  // configure the baseXORMasks
  numOnes = (int)(changeDegree * NGENES);
  for (i=0; i<totalBaseStates; ++i) {
    for (j=0; j<numOnes; ++j)
      baseXORMasks[i][randIndex[numOnes*i+j]] = 1; 
  }
}

/*******************************************************/
/* Environment Dynamics Function: Incremental changing */
/*******************************************************/
void changeEnvironment(void) {
  int i, j, numOnes, baseStateIdx;
  int intermMask[NGENES];  // intermediate mask

  for (i=0; i<NGENES; ++i) intermMask[i] = 0; 

  if ( bRandRatio == true)
    changeDegree = randNumber(0.01, 0.99);
  numOnes = (int)(changeDegree * NGENES);

  switch(changeMode) {
    case 1: // cyclical change
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i)  // copy the relevant base XORing mask 
        intermMask[i] = baseXORMasks[baseStateIdx][i];
      break;
    case 2: // partially cyclical change
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i) // copy the relevant base XORing mask
        intermMask[i] = baseXORMasks[baseStateIdx][i];

      for (i=0; i<NGENES; ++i)
        if (randNumber(0.0, 1.0) < pBaseMut)
          intermMask[i] = 1 - intermMask[i];
      break;
    case 3: // random non-cyclical change
      int index2[numOnes];
      randomSequence(index2, numOnes, NGENES);
      for (i=0; i<numOnes; ++i) // create numOnes 1's in intermMask[]
        intermMask[index2[i]] = 1;
      break;
  }

  for (i=0; i<NGENES; ++i) // integrate intermMask[] into xorMask[]
    xorMask[i] ^= intermMask[i];
}

/***************************************************************/
/* Report function: Reports progress of the simulation. Data   */
/* dumped into the output file are separated by " ".           */
/***************************************************************/
void report(void) {
  int idx = 0;
  double popBestFitness=0.0; // best population fitness
  for (int i = 0; i < POPSIZE; ++i) { // including memory
    if(popBestFitness < pop[i].fitness) {
      popBestFitness = pop[i].fitness;
      idx = i;
    }
  }

  fprintf(logPopBest, "%d ", (int)popBestFitness);
}

/*************************************************************/
/* Reports diversity of the population. Data dumped into the */
/*   output file are separated by " ".                       */
/*************************************************************/
void reportDiversity(void) {
  int i, j, k, hammingDist;
  double popDiversity = 0.0; 
  for (i = 0; i<POPSIZE; ++i) 
    for (j = 0; j<POPSIZE; ++j) {
      hammingDist = 0;
      for (k = 0; k<NGENES; ++k) {
        if (pop[i].gene[k] != pop[j].gene[k]) hammingDist += 1;
      }
      popDiversity += (double)hammingDist;
    }

  popDiversity /= (double)(NGENES*POPSIZE*(POPSIZE-1));

  fprintf(logPopDiv, "%4.3f ", popDiversity);
}

/**************************************************************/
/* Main function: Each generation involves selecting the best */
/* members, performing crossover & mutation and then          */
/* evaluating the resulting population, until the terminating */
/* condition is satisfied                                     */
/**************************************************************/
int main(int argc, char *argv[]) {
  // read in parameters
  gaMode = atoi(argv[1]);
  funcNo = atoi(argv[2]);
  changeMode = atoi(argv[3]);
  changeDegree = atof(argv[4]);
  changeSpeed = atoi(argv[5]);
  immigrateRatio = atof(argv[6]);
  //pBaseMut = atof(argv[7]);
  //randSeed = atoi(argv[8]);
  randSeed = atoi(argv[7]);
  srand(randSeed);

  if (changeDegree > 1.0) bRandRatio = true;
  else bRandRatio = false;

  if (changeMode == 1 || changeMode == 2) {
    if(changeDegree == 0.1)  totalBaseStates = 10;
    if(changeDegree == 0.2)  totalBaseStates = 5;
    if(changeDegree == 0.5)  totalBaseStates = 2;
    if(changeDegree == 1.0)  totalBaseStates = 1;
    constructBaseXORMasks();
  }

  bool recordDiversity = false;
  if (changeSpeed == 25 && immigrateRatio == 0.2 
       && pBaseMut == 0.05) 
    recordDiversity = true;

  char * filename = new char[60];
  sprintf(filename, "PopBest_GA%d_DUF%d_%d_%4.2f_%d_%4.2f.txt",
          gaMode, funcNo, changeMode, changeDegree, changeSpeed, immigrateRatio);

  if ((logPopBest = fopen(filename, "a")) == NULL) { exit(2); }
  fprintf(logPopBest, "%d ", randSeed);

  if (recordDiversity == true) {
    char* divfilename = new char[60];
    sprintf(divfilename, "PopDiv_GA%d_DUF%d_%d_%4.2f_%d_%4.2f.txt", 
          gaMode, funcNo, changeMode, changeDegree, changeSpeed, immigrateRatio);

    if ((logPopDiv = fopen(divfilename,"a")) == NULL) { exit(3); }
    fprintf(logPopDiv, "%d ", randSeed); 
  }

  cout << "The program GA" << gaMode <<" for DUF" << funcNo
       << " is running with random seed " << randSeed << endl;
  currentGenNo = 0;
  nextMemUpdateTime = rand()%6 + 5 + currentGenNo;
  initialize();

  // while (currentGenNo <= changeSpeed * totalChanges) {
  while (currentGenNo <= 5000) {
    evaluatePop();
    evaluateMemory();
    if (currentGenNo > 0) retrieveElite();

    if (gaMode == 2) updateGenePool();

    if (changeDetected == true) {
      switch(gaMode) {
        case 1: 
          retrieveMemory(); restart(); break;
        case 2:
          retrieveMemory(); break;
        default: break;
      }
    }

    if (currentGenNo == nextMemUpdateTime) {
      updateMemory(); 
      nextMemUpdateTime = rand()%6 + 5 + currentGenNo;
    }

    storeElite();
    report();
    if (recordDiversity == true) reportDiversity();

    select();
    switch(gaMode) {
      case 1:  
        crossover(0, popSize1);
        crossover(popSize1, popSize1+popSize2);
        mutate(pMutation); break;
      case 2:
        clone(); 
        mutate(pMutation); break;
      default: break;
    }

    if ((currentGenNo%changeSpeed) == 0) // change environment
      changeEnvironment();
    currentGenNo++;
  }

  fprintf(logPopBest, "\n");  // next line for next run
  fclose(logPopBest);
  if (recordDiversity == true) {
    fprintf(logPopDiv, "\n");  // next line for next run
    fclose(logPopDiv);
  }
  cout << "Success!" << endl;
  return 0;
}

/**************************** End *************************/
