/**************************************************************************************/
/* PBIL4DOP.c                                                                         */
/* This is an implementation of Population-Based Incremental Learning (PBIL)          */
/* algorithm for dynamic optimisation problems with the XOR generator for the         */
/* following paper:                                                                   */
/*                                                                                    */
/*   S. Yang and X. Yao. Population-based incremental learning with associative       */
/*   memory for dynamic environments. IEEE Transactions on Evolutionary Computation,  */
/*   12(5): 542-561, October 2008. IEEE Press (DOI: 10.1109/TEVC.2007.913070).        */
/*                                                                                    */
/* Compile:                                                                           */
/*   g++ -o PBIL4DOP PBIL4DOP.c                                                       */
/* Run:                                                                               */
/*   ./PBIL4DOP pbilMode funcNo chgMode chgDeg chgSpeed immRate lrnRate nElite1 rSeed */
/*                                                                                    */
/* Written by:                                                                        */
/*   Shengxiang Yang, University of Leicester, UK, May 2005; refined November 2009    */
/*                                                                                    */
/* If any query, please email Shengxiang Yang at s.yang@mcs.le.ac.uk                  */
/*                                                                                    */
/**************************************************************************************/

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string>
#include <fstream>

using namespace std; 

/* Change any of these parameters to match your needs */

#define NGENES 100   // no. of genes in a binary chromosome
#define POPSIZE 100  // total population size
#define MEMSIZE 10   // memory size
#define MAXBASESTATES 20  // total base states for cyclical changes

/* Global variables */
FILE *logPopBest;   // output file for best fitness in the pop
FILE *logPopDiv;    // output file for pop diversity
int currentGenNo=0; // current generation no
int randSeed = 0;
int pbilMode;
   // PBIL mode: 1, 2, 3, and 4 one pop, the others two pops
   // 1: SPBIL,  simple PBIL with 1 pop and no memory
   // 2: SPBILi, PBIL with 1 pop and random immigrants replacing worst samples
   // 3: MPBIL,  1 pop with mem, best sample replaces nearest mem point
   // 4: MPBILi, as 3 but with random immigrants replacing worst samples

   // 5: SPBIL2,  simple PBIL with 2 pops and no memory
   // 6: MPBIL2,  2 pops, both save into memory but only one retrieve from memory 
   // 7: MPBIL2r, as 6 but the one doing no memory retrieveal has restart scheme

int funcNo = 1;      // DUF Function: 1-DUF(25,4,1), 2-DUF(25,4,3), 3-DUF(25,4,4), 
                     //      4-DUF(25,4,deceptive), 25 copies of 4-bit deceptive problems
int changeMode = 1;  // how environment changes, cyclical or not
                     //   1: cyclical, 2: cyclical with noise, 3: non-cyclical (random)

double f_Ones1[NGENES]; // freq. of ones in each locus in elites of subpop1 (or pop)
double f_Ones2[NGENES]; // freq. of ones in each locus in elites of subpop2
double probVector1[NGENES]; // 1st prob. vector of generating ones in each locus
double probVector2[NGENES]; // 2nd (dual) prob. vector of generating ones in each locus
double probCentral[NGENES]; // central prob. vector, always generating ones with prob. 0.5 
double learnRate = 0.25;    // learning rate of PBILs

int nElite1;  // number of elites selected from subpop1 to learn from
int nElite2;  // number of elites selected from subpop2 to learn from
int popSize1;  // number of samples created by 1st prob. vector
int popSize2;  // number of samples created by 2nd prob. vector

int minPopSize1;  // min number of samples created by 1st prob. vector
int minPopSize2;  // min number of samples created by 2nd prob. vector
int maxPopSize1;  // max number of samples created by 1st prob. vector
int maxPopSize2;  // max number of samples created by 2nd prob. vector
int deltaPopSize; // amount of pop size to be adjusted for prob vectors

double changeDegree=0.5; // ratio of loci changed in dynamic environment
int changeSpeed = 10;    // period of environment changing in generations
int totalChanges = 100;  // total number of environment changes in a run
int xorMask[NGENES];     // mask for XORing environment
bool bRandRatio =false;  // whether the changeDegree is random (default to false)
int baseXORMasks[MAXBASESTATES][NGENES]; // base xor masks for cyclical environment
int totalBaseStates;
//double ratioPerturb = 0.1; // ratio of bits perturbed to the ones in a base XOR mask 

double pBaseMut = 0.05; // prob. of mutating each element of a base XOR mask 
double mutProb = 0.02;  // prob. of mutating each element of prob. vectors
double mutShift= 0.05;  // amount a mutation alters the value in each element

int memSize = (int)(0.1*POPSIZE);   // memory size
int memLevel;         // number of points in the memory (full if memLevel == memSize) 
int nextMemUpdateTime; // next memory update generation for MPBIL V MPBIL2, memory
                      // updated after next random number of gnerations in [5, 10]
double memProbVecs[MEMSIZE][NGENES]; // prob. vectors relevant to stored memory points 
bool changeDetected;  // whether the enviroment change has been detected
double immigrateRatio = 0.2; // ratio of random immigrants to total population size
int nImmigrants;             // number of random immigrants

struct genotype {        // genotype (GT), a member of the population
  int gene[NGENES];        
  int phenoGene[NGENES]; // phenotype allele for dynamic environment      
  double fitness;
  double oldfitness;     // fitness of previous generation for memory points
                         // used for detecting environmental changes 
};

struct genotype pop[POPSIZE];  // sample set (population), 
                               // pop[POPSIZ-memSize-1] to pop[POPSIZ-1] are 
                               // used as memory if the memory scheme is used
struct genotype newpop[POPSIZE]; // temporary population
struct genotype tmpMember;       // temporary individual for sortPopulation()


/***************************************************/
/* Utility Functions                               */
/***************************************************/
int randomBit() { return (rand()%2); }

double randNumber(double low, double high) {
  return ((double)(rand()%10000)/10000.0)*(high - low) + low;  
}

int sign(double value) {
  if (value == 0.0) return 0; 
  else return (value > 0.0? 1 : -1);
}

template <typename T>
void swap(T *x, T *y) {
  T temp;
  temp = *x; *x = *y; *y = temp;
}

/***********************************************************/
/* Random sequence generator:                              */
/*   Generates a random sequence of seqSize numbers from   */
/*   the natural number array [0, 1, ..., numSize-1]       */
/***********************************************************/
void randomSequence(int* seqArr, int seqSize, int numSize) {
  int i, j, idx, count, numleft;

  // constructs a natural number array
  int number[numSize];
  for (i = 0; i < numSize; ++i) number[i] = i; 

  // select seqSize numbers from number[] without repeat
  numleft = numSize;
  for (i = 0; i < seqSize; ++i) { 
    idx = rand() % numleft;
    count = 0;
    for (j = 0; j < numSize; ++j) { 
      if (number[j] != -1) // found one not selected number
        ++count;

      if (count > idx) break; // found the idx-th not selected number
    }

    seqArr[i] = number[j];
    number[j] = -1; // marked as selected
    --numleft;
  }
}

/******************************************************/
/* UF function                                        */
/******************************************************/
#define NBLOCKS 25    // the number of schemas
#define BLOCKSIZE 4   // block size - length of target schema

void DUF(int mem) {
  int i, j, countOFOnes;

  for (i = 0; i < NGENES; ++i) // perform XORing operation
    pop[mem].phenoGene[i] = pop[mem].gene[i] ^ xorMask[i];

  pop[mem].fitness = 0.0;
  for (i = 0; i < NBLOCKS; ++i) { 
    countOFOnes=0;
    for (j = 0; j < BLOCKSIZE; j++)
      if (pop[mem].phenoGene[i*BLOCKSIZE+j] == 1)
        countOFOnes++;  // count the 1 bits in the block

    switch(funcNo) {
      case 1: // DUF(25,4,1)
        pop[mem].fitness += countOFOnes; 
        break;
      case 2: // DUF(25,4,2)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else if(countOFOnes == 3) pop[mem].fitness += 2;
        break;
      case 3: // DUF(25,4,4)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        break;
      case 4: // DUF(25,4,deceptive)
        if (countOFOnes == 4) pop[mem].fitness += 4;
        else pop[mem].fitness += (3 - countOFOnes);
        break;
    }
  }
}

/*************************************************************/
/* Initialization function: Initializes the prob. vectors.   */
/*************************************************************/
void initialize() {
  int i, j, mem;

  // intitialize the XOR mask to static problem
  for (i = 0; i < NGENES; ++i) xorMask[i] = 0;

  for (mem = 0; mem <= POPSIZE; ++mem)
    for (i = 0; i <= NGENES; ++i) {
      pop[mem].gene[i] = 0;
      pop[mem].phenoGene[i] = 0;
    }

  for (i = 0; i < NGENES; ++i) {
    switch(pbilMode) {
      case 1:  // SPBIL
      case 2:  // SPBILi
      case 3:  // MPBIL
      case 4:  // MPBILi
       probVector1[i] = 0.5;
        break;
      case 5:  // SPBIL2
      case 6:  // MPBIL2
      case 7:  // MPBIL2r
        probVector1[i] = 0.5;
        probVector2[i] = randNumber(0.0, 1.0); 
        break;
    }
  }

  nElite1 = nElite2 = 1;
  switch(pbilMode) {
    case 1:  // SPBIL
    case 2:  // SPBILi
      popSize1 = POPSIZE;
      memSize = 0;
      break;
    case 3:  // MPBIL 
    case 4:  // MPBILi
      popSize1 = (int)(0.9*POPSIZE);
      memSize  = (int)(0.1*POPSIZE);

      memLevel = 0;
      for (i = 0; i < memSize; ++i) {
        for (j = 0; j < NGENES; ++j)
          memProbVecs[i][j] = 0.5;
      }
      break;
    case 5:  // SPBIL2
      popSize1 = (int)(0.5*POPSIZE);
      popSize2 = (int)(0.5*POPSIZE);
      memSize = 0;
      minPopSize1 = (int)(0.3*POPSIZE);
      minPopSize2 = (int)(0.3*POPSIZE);
      maxPopSize1 = (int)(0.7*POPSIZE);
      maxPopSize2 = (int)(0.7*POPSIZE);
      break;

    case 6:  // MPBIL2
    case 7:  // MPBIL2r
      popSize1 = (int)(0.45*POPSIZE);
      popSize2 = (int)(0.45*POPSIZE);
      memSize  = (int)(0.1*POPSIZE);
      minPopSize1 = (int)(0.3*POPSIZE);
      minPopSize2 = (int)(0.3*POPSIZE);
      maxPopSize1 = (int)(0.6*POPSIZE);
      maxPopSize2 = (int)(0.6*POPSIZE);

      memLevel = 0;
      for (i = 0; i < memSize; ++i) {
        for (j = 0; j < NGENES; ++j)
          memProbVecs[i][j] = 0.5;
      }
      break;
  }
  deltaPopSize = (int)(0.05*POPSIZE); 

  for (mem = 0; mem < POPSIZE; ++mem) {
    pop[mem].fitness = 0.0;
    pop[mem].oldfitness = 0.0;
  }

  nImmigrants = (int)(immigrateRatio * POPSIZE);
  changeDetected = false;
}

/***********************************************************/
/* Sort sub-population in the order of decreasing fitness  */
/*   using the bubble-sorting algorithm                    */
/***********************************************************/
void sortPopulation(int start, int end) {
  int i, j;
  for (i = start; i < end; ++i)
    for (j = start + 1; j < start + end - i; ++j)
      if (pop[j-1].fitness < pop[j].fitness) {
        tmpMember = pop[j-1];
        pop[j-1] = pop[j];
        pop[j] = tmpMember;
      }
}

/***********************************************************/
/* evaluate the main population, excluding memory if used  */
/***********************************************************/
void evaluatePop() {
  for (int mem = 0; mem < POPSIZE - memSize; ++mem) 
    DUF(mem);
}

/***********************************************************/
/* evaluate memory and detect environmental changes        */
/***********************************************************/
void evaluateMemory() {
  int i, memStartID = POPSIZE - memSize;
  for (i = memStartID; i < POPSIZE; ++i) // store old fitness
    pop[i].oldfitness = pop[i].fitness;

  changeDetected = false; 
  for (i = memStartID; i < POPSIZE; ++i) {
    DUF(i);

    // if not the initial generation, check whether change occurs
    if (currentGenNo != 0) {  
      if (pop[i].oldfitness != pop[i].fitness)
        changeDetected = true;
    }
  }
}

void storeElite() {
  int idx;
  switch(pbilMode) {
    case 1: case 2: case 3: case 4: 
      sortPopulation(0, popSize1);  // sort population
      idx = 0;
      break;
    case 5: case 6: case 7:
      sortPopulation(0, popSize1);  // sort population 1 
      sortPopulation(popSize1, popSize1+popSize2);  // sort population 2
      idx = (pop[0].fitness > pop[popSize1].fitness) ? 0 : popSize1;
      break;
    default: break;
  }

  pop[POPSIZE] = pop[idx]; // store elite
}

// replace the worst individual in pop 1 with stored elite
void retrieveElite() {
  sortPopulation(0, popSize1);  // sort population 1
  pop[popSize1-1] = pop[POPSIZE];

  DUF(popSize1-1); // re-evaluate the elite in case change occurs
}

/***************************************************************/
/* Sampling function: generate samples using the prob. vectors */
/***************************************************************/
void sample() {
  int i, j;
  double p;
  switch(pbilMode){
    case 1: case 2: case 3: case 4:
      for (i = 0; i < popSize1; ++i)
        for (j = 0; j < NGENES; ++j) {
          p = randNumber(0.0, 1.0);
          if (p < probVector1[j]) pop[i].gene[j] = 1;
          else pop[i].gene[j] = 0;
        };
      break;
    case 5: case 6: case 7:
      for (i = 0; i < popSize1; ++i)
        for (j = 0; j < NGENES; ++j) {
          p = randNumber(0.0, 1.0);
          if (p < probVector1[j]) pop[i].gene[j] = 1;
          else pop[i].gene[j] = 0;
        };
      for (i = popSize1; i < POPSIZE - memSize; ++i)
        for (j = 0; j < NGENES; ++j) {
          p = randNumber(0.0, 1.0);
          if (p < probVector2[j]) pop[i].gene[j] = 1;
          else pop[i].gene[j] = 0;
        };
      break;
  }
}

void immigrate(void) {
  int i, j, mem;
  double p;
  int memStartIdx = POPSIZE - memSize;

  sortPopulation(0, popSize1);  // sort pop 1 

  // now create random immigrants to replace worst ones
  switch(pbilMode) {
    case 2: case 4:
      for (mem = 0; mem < nImmigrants; ++mem) { // random immigrate
        for (i = 0; i < NGENES; ++i) {
          pop[popSize1-1-mem].gene[i] = randomBit();
          DUF(popSize1-1-mem); // evaluate the immigrant
	}
      }
      break; 
    default: break;
  }
}


/***********************************************************/
/* Learning function: learn the Prob. Vectors              */
/***********************************************************/
void learnProbVector() {
  int i, j, temp1, temp2;
  double tmpProb;
  double bestFitness1, bestFitness2;

  switch(pbilMode) { // obtain best elites
    case 1: case 2: case 3: case 4:
      sortPopulation(0, popSize1);
      break;
    case 5: case 6: case 7:
      sortPopulation(0, popSize1);
      sortPopulation(popSize1, POPSIZE - memSize);
      bestFitness1 = pop[0].fitness;
      bestFitness2 = pop[popSize1].fitness;
      break;
  }

  for (i = 0; i < NGENES; ++i) { // doing statistics from best elites
    temp1 = temp2 = 0;
    switch(pbilMode){
      case 1: case 2: case 3: case 4: 
        for (j = 0; j < nElite1; ++j)
          temp1 += pop[j].gene[i];
        f_Ones1[i] = (double)temp1/nElite1;
        break;
      case 5: case 6: case 7:
        for (j = 0; j < nElite1; ++j)
          temp1 += pop[j].gene[i];
        for (j = popSize1; j < popSize1 + nElite2; ++j)
          temp2 += pop[j].gene[i];
        f_Ones1[i] = (double)temp1/nElite1;
        f_Ones2[i] = (double)temp2/nElite2;
        break;
    }
  }

  for (i = 0; i < NGENES; ++i) { // learning from best elites in pop
    switch(pbilMode){
      case 1: case 2: case 3: case 4:
        tmpProb = (1.0 - learnRate)*probVector1[i] + learnRate*f_Ones1[i];
        probVector1[i] = tmpProb; // limitProb(tmpProb);
        break;
      case 5: case 6: case 7:
        tmpProb = (1.0 - learnRate)*probVector1[i] + learnRate*f_Ones1[i];
        probVector1[i] = tmpProb; // limitProb(tmpProb);
        tmpProb = (1.0 - learnRate)*probVector2[i] + learnRate*f_Ones2[i];
        probVector2[i] = tmpProb; // limitProb(tmpProb);
        break;
    }
  }

  switch(pbilMode) {  // adjust popsizes
    case 5: case 6: case 7:
      if( bestFitness1 > bestFitness2 )
        popSize1 = min(popSize1 + deltaPopSize, maxPopSize1);
      else if( bestFitness1 < bestFitness2 )
        popSize1 = max(popSize1 - deltaPopSize, minPopSize1);
      popSize2 = POPSIZE - popSize1 - memSize;
      break;
    default: break;
  }
}

/***********************************************************/
/* Mutation function: mutate the Prob. Vectors             */
/***********************************************************/
void mutateProbVector() {
  int i, j, temp1, temp2;
  double tmpProb;

  switch(pbilMode) {
    case 1: case 2: case 3: case 4:
      for (int i = 0; i < NGENES; ++i) {
        if ( randNumber(0.0, 1.0) < mutProb) {
	  tmpProb = (probVector1[i] > 0.5) ? ((1.0 - mutShift)*probVector1[i]) : 
	    ((1.0 - mutShift)*probVector1[i] + mutShift);
	  probVector1[i] = tmpProb; 
        }
      }
      break;
    case 5: case 6: case 7: 
      for (int i = 0; i < NGENES; ++i) {
        if ( randNumber(0.0, 1.0) < mutProb) {
	  tmpProb = (probVector1[i] > 0.5) ? ((1.0 - mutShift)*probVector1[i]) : 
	    ((1.0 - mutShift)*probVector1[i] + mutShift);
          probVector1[i] = tmpProb;
        }
      }

      for (int i = 0; i < NGENES; ++i) {
        if ( randNumber(0.0, 1.0) < mutProb) {
	  tmpProb = (probVector2[i] > 0.5) ? ((1.0 - mutShift)*probVector2[i]) : 
	    ((1.0 - mutShift)*probVector2[i] + mutShift);
	  probVector2[i] = tmpProb;
        }
      }
      break;
  }
}


/********************************************************/
/* The memory prob. vectors compete for the chance to   */
/* replace the pop prob. vector                         */
/********************************************************/
void competeProbVector() {
  int i, j, k, sampleSize;
  double p, maxMeanFitness, meanFitness; 
  double bestProbVector[NGENES];
  sampleSize = (int)POPSIZE/(memSize+1);

  // store the pop temporarily
  for (i = 0; i < POPSIZE; ++i) newpop[i] = pop[i];

  meanFitness = 0.0;
  for (i = 0; i < popSize1; ++i) 
    meanFitness += pop[i].fitness;
  meanFitness /= popSize1;

  for (i = 0; i < NGENES; ++i) 
    bestProbVector[i] = probVector1[i];
  maxMeanFitness = meanFitness;

  // each memory prob vector generates sampleSize samples and 
  // compete with the current best prob vector 
  for (i = 0; i < memLevel; ++i) {
    for (j = 0; j < sampleSize; ++j) {
      for (k = 0; k < NGENES; ++k) {
        p = randNumber(0.0, 1.0);
        if (p < memProbVecs[i][k]) pop[j].gene[k] = 1;
        else pop[j].gene[k] = 0;
      }
      DUF(j);
    }

    // calculate the mean fitness of samples of each prob. vector
    meanFitness = 0.0;
    for (j = 0; j < sampleSize; ++j) 
      meanFitness += pop[j].fitness;
    meanFitness /= sampleSize;
    
    // replace the current best prob vector if this one is better
    if (maxMeanFitness < meanFitness) {
      for (k = 0; k < NGENES; ++k) 
        bestProbVector[k] = memProbVecs[i][k];
      maxMeanFitness = meanFitness;
    }
  }

  // finally compete with the central prob vector
  for (j = 0; j < sampleSize; ++j) {
    for (k = 0; k < NGENES; ++k) 
      pop[j].gene[k] = ((randNumber(0.0, 1.0) > 0.5)? 0 : 1);

    DUF(j);
  }

  // calculate the mean fitness of samples of the central prob vector
  meanFitness = 0.0;
  for (j = 0; j < sampleSize; ++j) 
    meanFitness += pop[j].fitness;
  meanFitness /= sampleSize;

  // replace the current best prob vector if the central prob vector 
  // is better
  if (maxMeanFitness < meanFitness) {
    for (k = 0; k < NGENES; ++k)
      bestProbVector[k] = 0.5;
    maxMeanFitness = meanFitness;
  }

  // store the current best prob vector as the pop prob vector
  for (k = 0; k < NGENES; ++k)
    probVector1[k] = bestProbVector[k];

  // copy the original pop back
  for (i = 0; i < POPSIZE; ++i)
    pop[i] = newpop[i];
}

/*************************************************************/
/* Restart a prob. vector when environment change is deteced */
/*************************************************************/
void restart() {
  if(pbilMode==7) { // MPBIL2r
    for (int i=0; i<NGENES; ++i)
      probVector2[i] = 0.5;
  }
}

/***********************************************************/
/* Update prob. vectors when environment change is deteced */
/***********************************************************/
void updateProbVector() {
  // do only when memory is not empty, other skip
  if (memLevel > 0) {
    int i, bestMemIdx, secondBestMemIdx, memStartIdx;
    double maxFitness;
    memStartIdx = POPSIZE - memSize;

    switch(pbilMode) {  // sort populations
      case 3:  // MPBIL
      case 4:  // MPBILi
        sortPopulation(0, popSize1);
        break;
      case 6:  // MPBIL2
      case 7:  // MPBIL2r
        sortPopulation(0, popSize1);
        sortPopulation(popSize1, popSize1+popSize2);
        break;
      default: break;
    }

    // find the best memory point index
    bestMemIdx = 0;
    maxFitness = pop[memStartIdx].fitness;
    for (i = 1; i < memLevel; ++i) 
      if(pop[memStartIdx+i].fitness > maxFitness) {
        maxFitness = pop[memStartIdx+i].fitness;
        bestMemIdx = i;
      }

    switch(pbilMode) {  // update the pop prob vector 
      case 3:  // MPBIL
      case 4:  // MPBILi
      case 6:  // MPBIL2
      case 7:  // MPBIL2r
        // update when memory is better
        if (pop[memStartIdx+bestMemIdx].fitness > pop[0].fitness)
          for (i = 0; i < NGENES; ++i)  
            probVector1[i] = memProbVecs[bestMemIdx][i];
        break;
    }
  }
}

/*******************************************************/
/* Retrieve best individual and prob. vector from main */
/* population to replace closest memory point and its  */ 
/* prob. vector                                        */
/*******************************************************/
void updateMemory(void) {
  int i, j;
  int tmpDist, minHammingDist;
  double tmpProbVecDist, minProbVecDist;
  int memStartIdx = POPSIZE - memSize;
  int index;          // index of the memory point to be changed
  int bestSampleIdx;  // index of the best sample in pop

  switch(pbilMode) {  // sort the pop
    case 3:  // MPBIL
    case 4:  // MPBILi
      sortPopulation(0, popSize1);
      bestSampleIdx = 0;
      break;
    case 6:  // MPBIL2
    case 7:  // MPBIL2r
      sortPopulation(0, popSize1);
      sortPopulation(popSize1, popSize1+popSize2);
      if(pop[0].fitness >= pop[popSize1].fitness)
        bestSampleIdx = 0;        // prob vector 1 is better
      else
        bestSampleIdx = popSize1; // prob vector 2 is better
      break;
  }

  // if memory not full, store as new point; otherwise, replace the closest
  if (memLevel < memSize) {
    pop[memStartIdx + memLevel] = pop[bestSampleIdx];
    for (i = 0; i < NGENES; ++i)
      memProbVecs[memLevel][i] =
         (bestSampleIdx == 0 ? probVector1[i]: probVector2[i]);

    memLevel++;
  }
  else {  // memory is full
    index = 0;
    // find the memory point closest to the best sample of pop
    minHammingDist = NGENES;
    for (i = 0; i < memSize; ++i) {
      tmpDist = 0;
      for (j = 0; j < NGENES; ++j) { // calculate the Hamming distance
        if(pop[i + memStartIdx].gene[j] != pop[bestSampleIdx].gene[j]) 
          tmpDist++;
      }
      if(tmpDist < minHammingDist) { 
        minHammingDist = tmpDist; 
        index = i;
      }
    }
   
    // update closest memory point if the best sample of pop is better
    if( pop[memStartIdx+index].fitness < pop[bestSampleIdx].fitness) {
      pop[index + memStartIdx] = pop[bestSampleIdx];
      if (bestSampleIdx == 0) 
        for (i = 0; i < NGENES; ++i)
          memProbVecs[index][i] = probVector1[i];
      else 
        for (i = 0; i < NGENES; ++i)
          memProbVecs[index][i] = probVector2[i];      
    }
  }
}

/***************************************************************/
/* Report function: Reports progress of the simulation. Data   */
/* dumped into the output file are separated by " ".           */
/***************************************************************/
void report(void) {
  double popBestFitness = 0.0; // best population fitness
  for (int i = 0; i < POPSIZE; ++i)  // memory inclusive
    if(popBestFitness < pop[i].fitness)
      popBestFitness = pop[i].fitness;

  fprintf(logPopBest, "%d ", (int)popBestFitness);
}

/*********************************************************/
/* Environment Dynamics Function: Construct base XORing  */
/* masks for (partially) cyclical changing environments  */
/*********************************************************/
void constructBaseXORMasks(void) {
  int i, j, numOnes;
  int randIndex[NGENES];

  // create a random permutation of 0 to NGENES-1
  randomSequence(randIndex, NGENES, NGENES);

  // initialize the baseXORMasks
  for (i=0; i<totalBaseStates; ++i)
    for (j=0; j<NGENES; ++j) baseXORMasks[i][j] = 0;

  // configure the baseXORMasks
  numOnes = (int)(changeDegree * NGENES);
  for (i=0; i<totalBaseStates; ++i) {
    for (j=0; j<numOnes; ++j)
      baseXORMasks[i][randIndex[numOnes*i+j]] = 1; 
  }
}

/*******************************************************/
/* Environment Dynamics Function: Incremental changing */
/*******************************************************/
void changeEnvironment(void) {
  int i, j, numOnes, baseStateIdx;
  int intermMask[NGENES];  // intermediate mask

  for (i=0; i<NGENES; ++i) intermMask[i] = 0; 

  if ( bRandRatio == true)
    changeDegree = randNumber(0.01, 0.99);
  numOnes = (int)(changeDegree * NGENES);

  switch(changeMode) {
    case 1: // cyclical change
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i)  // copy the relevant base XORing mask 
        intermMask[i] = baseXORMasks[baseStateIdx][i];
      break;
    case 2: // cyclical change with noise
      baseStateIdx = ((int)(currentGenNo/changeSpeed))%totalBaseStates;
      for (i=0; i<NGENES; ++i) // copy the relevant base XORing mask
        intermMask[i] = baseXORMasks[baseStateIdx][i];

      for (i=0; i<NGENES; ++i)
        if (randNumber(0.0, 1.0) < pBaseMut)
          intermMask[i] = 1 - intermMask[i];
      break;
    case 3: // random non-cyclical change
      int index2[numOnes];
      randomSequence(index2, numOnes, NGENES);
      for (i=0; i<numOnes; ++i) // create numOnes 1's in intermMask[]
        intermMask[index2[i]] = 1;
      break;
  }

  for (i=0; i<NGENES; ++i) // integrate intermMask[] into xorMask[]
    xorMask[i] ^= intermMask[i];
}

/*************************************************************/
/* Reports diversity of the population. Data dumped into the */
/*   output file are separated by " ".                       */
/*************************************************************/
void reportDiversity(void) {
  int i, j, k, hammingDist;
  double popDiversity = 0.0; 
  for (i = 0; i<POPSIZE; ++i) 
    for (j = 0; j<POPSIZE; ++j) {
      hammingDist = 0;
      for (k = 0; k<NGENES; ++k) {
        if (pop[i].gene[k] != pop[j].gene[k]) hammingDist += 1;
      }
      popDiversity += (double)hammingDist;
    }

  popDiversity /= (double)(NGENES*POPSIZE*(POPSIZE-1));

  fprintf(logPopDiv, "%4.3f ", popDiversity);
}

/**************************************************************/
/* Main function: Each iteration involves sampling the        */
/* probability vector(s), evaluating and selecting the best   */
/* sample(s), learning the probability vector(s) toward best  */
/* sample(s), (and mutating the probability vector(s)), until */
/* the terminating condition is satisfied                     */
/**************************************************************/
int main(int argc, char *argv[]) {
  pbilMode = atoi(argv[1]);
  funcNo = atoi(argv[2]);
  changeMode = atoi(argv[3]);
  changeDegree = atof(argv[4]);
  if (changeDegree > 1.0) bRandRatio = true;
  else bRandRatio = false;
  changeSpeed = atoi(argv[5]);
  immigrateRatio = atof(argv[6]);
  learnRate = atof(argv[7]);
  nElite1 = atoi(argv[8]);
  randSeed = atoi(argv[9]);
  srand(randSeed);

  if (changeMode == 1 || changeMode == 2) {
    if(changeDegree == 0.1)  totalBaseStates = 10;
    if(changeDegree == 0.2)  totalBaseStates = 5;
    if(changeDegree == 0.5)  totalBaseStates = 2;
    if(changeDegree == 1.0)  totalBaseStates = 1;
    constructBaseXORMasks();
  }

  bool recordDiversity = false;
  if (changeSpeed == 25 && immigrateRatio == 0.2
      && learnRate == 0.25 && nElite1 == 1)
    recordDiversity = true;

  char * filename = new char[60];
  sprintf(filename, "PopBest_PBIL%d_DUF%d_%d_%4.2f_%d_%4.2f_%4.2f_%d.txt", 
          pbilMode, funcNo, changeMode, changeDegree, changeSpeed, 
          immigrateRatio, learnRate, nElite1);

  if ((logPopBest = fopen(filename, "a")) == NULL) { exit(2); }
  fprintf(logPopBest, "%d ", randSeed);

  if (recordDiversity == true) {
    char* divfilename = new char[60];
    sprintf(divfilename, "PopDiv_PBIL%d_DUF%d_%d_%4.2f_%d_%4.2f_%4.2f_%d.txt", 
            pbilMode, funcNo, changeMode, changeDegree, changeSpeed, 
            immigrateRatio, learnRate, nElite1);

    if ((logPopDiv = fopen(divfilename, "a")) == NULL) { exit(3); }
    fprintf(logPopDiv, "%d ", randSeed); 
  }

  cout << "The program PBIL" << pbilMode <<" for DUF" << funcNo
       << " is running with random seed " << randSeed << endl;
  currentGenNo = 0;
  nextMemUpdateTime = rand()%6 + 5 + currentGenNo;
  initialize();
  sample();

  // while( currentGenNo <= changeSpeed * totalChanges ) {
  while (currentGenNo <= 5000) {
    evaluatePop();
    switch(pbilMode) { // PBILs with memory
      case 3: case 4: case 6: case 7:
        evaluateMemory();
        break;
      default: break;
    }

    switch(pbilMode) { // PBILs with random immigrants
      case 2: case 4: 
        immigrate();
        break;
      default: break;
    }

    // storeElite(); // elitism switched off
    report();
    if (recordDiversity == true) reportDiversity();

    if (currentGenNo == nextMemUpdateTime) { // time to update memory
      switch(pbilMode) { // PBILs with memory
        case 3: case 4: case 6: case 7: 
          // if change occurs as well (rarely), do not change memory becuase 
          // the best pop member is not associated with the pop prob. vector
          if (changeDetected == false) updateMemory(); 
          nextMemUpdateTime = rand()%6 + 5 + currentGenNo;
          break;
        default: break;
      }
    }

    if (changeDetected == true) {
      switch(pbilMode) {
        case 3:  // MPBIL
        case 4:  // MPBILi
        case 6:  // MPBIL2
          updateProbVector();
          break;
        case 7:  // MPBIL2r
          updateProbVector();
          restart();
          break;
        default: break;
      }
    }
    else learnProbVector();

    mutateProbVector();
    sample();

    if (currentGenNo%changeSpeed == 0)
      changeEnvironment();
    currentGenNo++;
  }

  fprintf(logPopBest, "\n");  // next line for next run
  fclose(logPopBest);
  if (recordDiversity == true) {
    fprintf(logPopDiv, "\n");  // next line for next run
    fclose(logPopDiv);
  }
  cout << "Success!" << endl;
  return 0;
}

/**************************** End ****************************/
